<?php

class tallerModel extends Model
{
    public $IdOrden;
    public $TipoOrden;
    public $FechaInicio;
    public $FechaTermino;
    public $IdUsuarioAccion;
    public $Motivo;
    public $Comentario;

    private function now_mx_sql(){ return "TIMESTAMPADD(HOUR, -6, UTC_TIMESTAMP())"; }

    // Órdenes en taller (EstadoTransito = 3), para TipoOrden=2 o 8
    public function taller_ordenes_en_taller_model(){
        if (empty($this->TipoOrden)) $this->TipoOrden = 2;
        $sql = "SELECT 
                    o.IdOrden, o.FolioOrden, o.IdCliente, o.IdDomicilio,
                    o.Fecha, o.TipoPago, o.Estatus, o.EstadoTransito, o.TipoOrden,
                    to2.descripcion AS TipoOrdenNombre,
                    tp.Descripcion AS TipoPagoNombre,
                    c.Nombre AS Cliente,
                    TRIM(CONCAT_WS(' ',
                        d.EntCalle, d.EntNExt, d.EntNInt, d.EntColonia,
                        d.EntCP, d.EntMunicipio, d.EntEstado
                    )) AS Domicilio,
                    g.idTallerGuia, g.estatus AS estatus_taller,
                    g.ts_inicio, g.ts_fin
                FROM ordenes o
                LEFT JOIN tipos_pago      tp  ON tp.IdPago   = o.TipoPago
                LEFT JOIN clientes        c   ON c.IdCliente   = o.IdCliente
                LEFT JOIN domentregacltes d   ON d.IdDomicilio = o.IdDomicilio
                LEFT JOIN tipo_orden      to2 ON to2.id        = o.TipoOrden
                LEFT JOIN taller_guias    g   ON g.IdOrden     = o.IdOrden
                WHERE o.EstadoTransito = 3
                  AND o.TipoOrden = :tipo
                ORDER BY o.Fecha DESC, o.IdOrden DESC";
        return parent::query($sql, ['tipo'=>$this->TipoOrden]) ?: [];
    }

    // Resguardo (EstadoTransito = 32)
    public function taller_resguardo_list_model(){
        $sql = "SELECT 
                    o.IdOrden, o.FolioOrden, o.IdCliente, o.IdDomicilio,
                    o.Fecha, o.TipoPago, o.Estatus, o.EstadoTransito, o.TipoOrden,
                    to2.descripcion AS TipoOrdenNombre,
                    c.Nombre AS Cliente,
                    TRIM(CONCAT_WS(' ',
                        d.EntCalle, d.EntNExt, d.EntNInt, d.EntColonia,
                        d.EntCP, d.EntMunicipio, d.EntEstado
                    )) AS Domicilio,
                    g.idTallerGuia, g.ts_resguardo_inicio, g.motivo_resguardo, g.comentario_resguardo
                FROM ordenes o
                LEFT JOIN clientes        c   ON c.IdCliente   = o.IdCliente
                LEFT JOIN domentregacltes d   ON d.IdDomicilio = o.IdDomicilio
                LEFT JOIN tipo_orden      to2 ON to2.id        = o.TipoOrden
                LEFT JOIN taller_guias    g   ON g.IdOrden     = o.IdOrden
                WHERE o.EstadoTransito = 32
                ORDER BY g.ts_resguardo_inicio DESC, o.IdOrden DESC";
        return parent::query($sql, []) ?: [];
    }

    // Iniciar trabajo (acepta orden + crea/actualiza guía + evento)
    public function taller_iniciar_trabajo_model(){
        if (empty($this->IdOrden)) throw new Exception("IdOrden requerido");
        $now = $this->now_mx_sql();

        // Determinar tipo desde orden
        $tipo = parent::query("SELECT TipoOrden FROM ordenes WHERE IdOrden=:id LIMIT 1", ['id'=>$this->IdOrden]);
        $tipoOrden = $tipo ? (int)$tipo[0]['TipoOrden'] : 2;

        // Marcar aceptación por taller
        parent::query("UPDATE ordenes SET AceptarOrdenTaller = 1 WHERE IdOrden = :id", ['id'=>$this->IdOrden]);

        // Upsert guía
        $sql = "INSERT INTO taller_guias
                    (IdOrden, TipoOrden, estatus, ts_inicio, created_by, updated_by, updated_at)
                VALUES
                    (:id, :tipo, 'EN_PROCESO', {$now}, :actor, :actor, {$now})
                ON DUPLICATE KEY UPDATE
                    TipoOrden = VALUES(TipoOrden),
                    estatus   = 'EN_PROCESO',
                    ts_inicio = COALESCE(taller_guias.ts_inicio, VALUES(ts_inicio)),
                    updated_by= VALUES(updated_by),
                    updated_at= VALUES(updated_at)";
        parent::query($sql, [
            'id'=>$this->IdOrden,
            'tipo'=>$tipoOrden,
            'actor'=>$this->IdUsuarioAccion ?: 0
        ]);

        // Evento de inicio
        $sqlE = "INSERT INTO taller_eventos (idTallerGuia, estatus, ts, nota)
                 SELECT g.idTallerGuia, 'INICIO_TALLER', {$now}, 'Inicio de trabajo'
                 FROM taller_guias g WHERE g.IdOrden = :id";
        parent::query($sqlE, ['id'=>$this->IdOrden]);

        return true;
    }

    // Terminar orden (marca ts_fin, guarda duración + evento)
    public function taller_terminar_orden_model(){
        if (empty($this->IdOrden)) throw new Exception("IdOrden requerido");
        $now = $this->now_mx_sql();

        $sql = "UPDATE taller_guias
                SET ts_fin = {$now},
                    duracion_seg = CASE WHEN ts_inicio IS NULL THEN NULL
                                        ELSE TIMESTAMPDIFF(SECOND, ts_inicio, {$now}) END,
                    estatus = 'TERMINADA',
                    updated_by = :actor,
                    updated_at = {$now}
                WHERE IdOrden = :id";
        parent::query($sql, ['id'=>$this->IdOrden, 'actor'=>$this->IdUsuarioAccion?:0]);

        $sqlE = "INSERT INTO taller_eventos (idTallerGuia, estatus, ts, nota)
                 SELECT g.idTallerGuia, 'TERMINADA', {$now}, 'Orden terminada en taller'
                 FROM taller_guias g WHERE g.IdOrden = :id";
        parent::query($sqlE, ['id'=>$this->IdOrden]);

        return true;
    }

    // Enviar a resguardo (actualiza orden + marca resguardo en guía + evento)
    public function taller_enviar_resguardo_model(){
        if (empty($this->IdOrden)) throw new Exception("IdOrden requerido");
        $now = $this->now_mx_sql();

        // EstadoTransito = 32
        parent::query("UPDATE ordenes SET EstadoTransito = 32 WHERE IdOrden = :id", ['id'=>$this->IdOrden]);

        // Guía en resguardo
        $sql = "INSERT INTO taller_guias
                    (IdOrden, TipoOrden, estatus, en_resguardo, ts_resguardo_inicio,
                     motivo_resguardo, comentario_resguardo, created_by, updated_by, updated_at)
                VALUES
                    (:id,
                     (SELECT o.TipoOrden FROM ordenes o WHERE o.IdOrden=:id LIMIT 1),
                     'RESGUARDO', 1, {$now}, :motivo, :coment, :actor, :actor, {$now})
                ON DUPLICATE KEY UPDATE
                    estatus = 'RESGUARDO',
                    en_resguardo = 1,
                    ts_resguardo_inicio = {$now},
                    motivo_resguardo = :motivo,
                    comentario_resguardo = :coment,
                    updated_by = :actor,
                    updated_at = {$now}";
        parent::query($sql, [
            'id'=>$this->IdOrden,
            'motivo'=>$this->Motivo ?: '',
            'coment'=>$this->Comentario ?: '',
            'actor'=>$this->IdUsuarioAccion ?: 0
        ]);

        // Evento
        $sqlE = "INSERT INTO taller_eventos (idTallerGuia, estatus, ts, nota)
                 SELECT g.idTallerGuia, 'RESGUARDO', {$now}, CONCAT('Resguardo: ', :motivo)
                 FROM taller_guias g WHERE g.IdOrden = :id";
        parent::query($sqlE, ['id'=>$this->IdOrden, 'motivo'=>$this->Motivo ?: '']);

        return true;
    }

    // Resumen métricas (TipoOrden=2)
    public function taller_resumen_metrics_model(){
    // Rango seleccionado (para agrupar por mes de CREACIÓN)
    $iniDate = !empty($this->FechaInicio) ? $this->FechaInicio : null;
    $finDate = !empty($this->FechaTermino) ? $this->FechaTermino : null;

    // Para filtros por ts_fin (otras métricas), usamos rango con tiempo
    $ini = $iniDate ? $iniDate.' 00:00:00' : null;
    $fin = $finDate ? $finDate.' 23:59:59' : null;

    // ---------- (A) KPI "Órdenes/día" -> % cierre en su mismo mes ----------
    // Denominador: órdenes creadas (Tipo 2) dentro del rango por FECHA de creación
    $paramsMes = [];
    $filtroCre = "";
    if ($iniDate && $finDate){
        $filtroCre = " AND DATE(o.Fecha) BETWEEN :iniD AND :finD ";
        $paramsMes = ['iniD'=>$iniDate, 'finD'=>$finDate];
    }

    $sqlTotCreadas = "SELECT COUNT(*) AS total_creadas
                      FROM ordenes o
                      WHERE o.TipoOrden = 2
                      {$filtroCre}";
    $rCre = parent::query($sqlTotCreadas, $paramsMes);
    $total_creadas = (int)($rCre[0]['total_creadas'] ?? 0);

    // Numerador: de esas mismas órdenes, solo las TERMINADAS cuyo ts_fin cae en el MISMO mes/año que su creación.
    // Importante: si se cerraron en otro mes, NO cuentan (ni para el mes de creación ni para el siguiente).
    $sqlCerradasMismoMes = "SELECT COUNT(*) AS total_cerradas
                            FROM ordenes o
                            INNER JOIN taller_guias g ON g.IdOrden = o.IdOrden
                            WHERE o.TipoOrden = 2
                              AND g.estatus   = 'TERMINADA'
                              AND g.ts_fin IS NOT NULL
                              {$filtroCre}
                              AND YEAR(g.ts_fin)  = YEAR(o.Fecha)
                              AND MONTH(g.ts_fin) = MONTH(o.Fecha)";
    $rCerr = parent::query($sqlCerradasMismoMes, $paramsMes);
    $cerr_mismo_mes = (int)($rCerr[0]['total_cerradas'] ?? 0);

    // Porcentaje 0–100 con 2 decimales (el front lo muestra con toFixed(2))
    $pct_cierre_mes = $total_creadas > 0 ? round(($cerr_mismo_mes / $total_creadas) * 100, 2) : 0.00;

    // ---------- (B) Otras métricas (se quedan como estaban) ----------
    // Terminadas y promedio de duración: filtran por ts_fin dentro del rango
    $paramsFin = [];
    $filtroFin = "";
    if ($ini && $fin){
        $filtroFin = " AND g.ts_fin BETWEEN :ini AND :fin ";
        $paramsFin = ['ini'=>$ini, 'fin'=>$fin];
    }

    $sql1 = "SELECT 
                COUNT(*) AS total_terminadas,
                ROUND(AVG(TIMESTAMPDIFF(SECOND, g.ts_inicio, g.ts_fin))/3600, 2) AS prom_horas
             FROM taller_guias g
             WHERE g.TipoOrden = 2
               AND g.estatus   = 'TERMINADA'
               {$filtroFin}";
    $r1 = parent::query($sql1, $paramsFin);
    $total_terminadas = (int)($r1[0]['total_terminadas'] ?? 0);
    $prom_horas       = (float)($r1[0]['prom_horas'] ?? 0);

    // Promedio de extintores por orden (solo terminadas en el rango por ts_fin)
    $sql2 = "SELECT ROUND(AVG(cnt), 2) AS prom_ext
             FROM (
                SELECT od.IdOrden, SUM(od.Cantidad) AS cnt
                FROM ordenesdetalle od
                INNER JOIN productos p ON p.proId = od.IdProducto AND p.proCatId = 1
                INNER JOIN taller_guias g ON g.IdOrden = od.IdOrden
                WHERE g.TipoOrden = 2
                  AND g.estatus   = 'TERMINADA'
                  {$filtroFin}
                GROUP BY od.IdOrden
             ) t";
    $r2 = parent::query($sql2, $paramsFin);
    $prom_ext = (float)($r2[0]['prom_ext'] ?? 0);

    // Activas en taller por tramos (situación actual)
    $now = $this->now_mx_sql();
    $sql3 = "SELECT 
                SUM(CASE WHEN TIMESTAMPDIFF(HOUR, g.ts_inicio, {$now}) < 24 THEN 1 ELSE 0 END) AS lt24,
                SUM(CASE WHEN TIMESTAMPDIFF(HOUR, g.ts_inicio, {$now}) BETWEEN 24 AND 47 THEN 1 ELSE 0 END) AS b24_48,
                SUM(CASE WHEN TIMESTAMPDIFF(HOUR, g.ts_inicio, {$now}) >= 48 THEN 1 ELSE 0 END) AS gt48
             FROM taller_guias g
             INNER JOIN ordenes o ON o.IdOrden = g.IdOrden
             WHERE o.EstadoTransito = 3
               AND g.ts_inicio IS NOT NULL
               AND g.ts_fin    IS NULL
               AND g.TipoOrden = 2";
    $r3 = parent::query($sql3, []);
    $lt24   = (int)($r3[0]['lt24'] ?? 0);
    $b24_48 = (int)($r3[0]['b24_48'] ?? 0);
    $gt48   = (int)($r3[0]['gt48'] ?? 0);

    // Devolvemos el porcentaje en la misma clave que usa el front
    return [
        'prom_ordenes_dia'          => $pct_cierre_mes,   // ← ahora es % 0–100 (el UI mostrará 2 decimales)
        'prom_extintores_por_orden' => $prom_ext,
        'prom_duracion_horas'       => $prom_horas,
        'activas_lt24'              => $lt24,
        'activas_24_48'             => $b24_48,
        'activas_gt48'              => $gt48,
        'terminadas'                => $total_terminadas   // sigue contando por ts_fin en el rango
    ];
}

}

