/* ============== caja.js COMPLETO (filtrado por sesión activa en TODOS los fetch) ============== */

(() => {
  // ===== Config =====
  const API_BASE = 'https://grupoxtinfire.com/admin/ajax/';       // AJUSTA si cambia
  const HOOK = 'codi_hook';
  const ACTION = 'load';
  // Antes: 'https://grupoxtinfire.com/admin/caja/pdfticket_cierre.php?idSesion='
  const PDF_CIERRE_URL = 'https://grupoxtinfire.com/admin/pdfreportes/pdfcierre_caja/';


  const ROOT_ID = 'caja-card-body';
  const $ = (sel, ctx=document) => ctx.querySelector(sel);
  const $$ = (sel, ctx=document) => Array.from(ctx.querySelectorAll(sel));

  // ===== Helpers =====
  const currency = new Intl.NumberFormat('es-MX', { style:'currency', currency:'MXN' });
  const fmt = (n)=> currency.format(Number(n||0));
  const theme = ()=> (document.documentElement.getAttribute('data-bs-theme') || 'light');
  const colorswal = (typeof colortema === 'function') ? colortema() : 'default';

  const isSupervisorDept = (dpt) => [1,16,17,24].includes(Number(dpt));
  const todayRange = () => {
    const now = new Date();
    const y = now.getFullYear();
    const m = String(now.getMonth()+1).padStart(2,'0');
    const d = String(now.getDate()).padStart(2,'0');
    const ini = `${y}-${m}-${d} 00:00:00`;
    const finDate = new Date(now.getFullYear(), now.getMonth(), now.getDate()+1);
    const y2 = finDate.getFullYear();
    const m2 = String(finDate.getMonth()+1).padStart(2,'0');
    const d2 = String(finDate.getDate()).padStart(2,'0');
    const fin = `${y2}-${m2}-${d2} 00:00:00`;
    return {ini, fin, dia:`${y}-${m}-${d}`};
  };

  // Toast
  const Toast = Swal.mixin({
    toast: true,
    position: "top-end",
    showConfirmButton: false,
    timer: 3000,
    timerProgressBar: true,
    didOpen: (t) => { t.onmouseenter = Swal.stopTimer; t.onmouseleave = Swal.resumeTimer; }
  });

  // POST helper
  async function apiPost(fnName, params={}) {
    const body = new URLSearchParams({ hook: HOOK, action: ACTION, ...params });
    const res = await fetch(API_BASE + fnName, {
      method: 'POST',
      headers: { 'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8' },
      body
    });
    if (!res.ok) throw new Error(`HTTP ${res.status} @${fnName}`);
    const json = await res.json();
    if (!json || json.status !== 200 || json.error) {
      throw new Error(json?.data || json?.msg || `API ${fnName} error`);
    }
    return json.data || [];
  }

  // ===== Pago normalization for expected totals =====
  function normalizePago(v) {
    const s = (v ?? '')
      .toString()
      .normalize('NFD').replace(/[\u0300-\u036f]/g,'')
      .trim().toUpperCase();

    if (s === '01' || s.includes('EFECTIVO'))                           return 'Efectivo';
    if (s === '03' || s.includes('TRANSFER'))                           return 'Transferencia';
    if (s === '04' || (s.includes('TARJETA') && s.includes('CREDIT')))  return 'Tarjeta Crédito';
    if (s === '28' || (s.includes('TARJETA') && s.includes('DEBIT')))   return 'Tarjeta Débito';
    if (s === '02' || s.includes('CHEQUE'))                             return 'Cheque';
    if (s === '99' || s.includes('SIN') || s.includes('POR DEFINIR'))   return 'SinTipodePago';
    return 'Pago desconocido';
  }
  function addTo(dict, key, val) {
    dict[key] = (Number(dict[key] || 0) + Number(val || 0));
  }

  // Calcula esperado (órdenes 11/12 + manuales ENTRADA−SALIDA + saldo inicial en Efectivo)
  function computeExpectedTotals() {
    const expected = {
      'Efectivo': 0, 'Transferencia': 0, 'Tarjeta Crédito': 0,
      'Tarjeta Débito': 0, 'Cheque': 0, 'SinTipodePago': 0, 'Pago desconocido': 0
    };

    // ÓRDENES que cuentan a caja (ya vienen filtradas por sesión)
    (state.ordenes || [])
      .filter(o => o.categoria === 'CAJA')
      .forEach(o => {
        const k = normalizePago(o.fid_pago ?? o.tipo_pago);
        addTo(expected, k, o.total);
      });

    // MOVIMIENTOS manuales netos (ya por sesión)
    (state.movs || []).forEach(m => {
      const signo = (m.tipo === 'ENTRADA') ? 1 : (m.tipo === 'SALIDA' ? -1 : 0);
      if (!signo) return;
      const k = normalizePago(m.TipoPago ?? m.tipo_pago);
      addTo(expected, k, signo * Number(m.monto || 0));
    });

    // SALDO INICIAL → Efectivo
    const saldoIni = Number(state.session?.saldo_inicial || 0);
    if (saldoIni) addTo(expected, 'Efectivo', saldoIni);

    return expected;
  }

  // ===== UI =====
  function buildUI(root) {
    root.innerHTML = `
      <div class="d-flex flex-column gap-3">

        <!-- Banner discrepancia -->
        <div id="caja-discrepancia-banner"></div>

        <!-- Header -->
        <div class="card">
          <div class="card-body">
            <div class="d-flex flex-wrap justify-content-between align-items-center gap-2">
              <div>
                <div class="small text-muted">Sesión actual</div>
                <div id="caja-session-banner" class="fw-semibold">—</div>
                <div id="caja-prevday-hint" class="mt-1"></div>
              </div>
              <div class="d-flex flex-wrap gap-2">
                <button id="btn-open-caja" type="button" class="btn btn-danger">Abrir caja</button>
                <button id="btn-manual-mov" type="button" class="btn btn-outline-danger">Movimiento manual</button>
                <button id="btn-solicitar-cierre" type="button" class="btn btn-dark">Solicitar cierre</button>
                <button id="btn-ver-solicitudes" type="button" class="btn btn-outline-danger">Solicitudes</button>
                <button id="btn-reimprimir-ticket" type="button" class="btn btn-outline-danger">Reimprimir último cierre</button>
                <button id="btn-discrepancias-mes" type="button" class="btn btn-outline-danger" style="display:none;">Discrepancias (mes)</button>
              </div>
            </div>
          </div>
        </div>

        <!-- KPIs -->
        <div id="caja-kpis" class="row g-3">
          <div class="col-12 col-md-4">
            <div class="card h-100">
              <div class="card-body">
                <div class="small text-muted">Saldo inicial</div>
                <div id="kpi-saldo-inicial" class="fs-4 fw-bold">—</div>
                <small id="kpi-saldo-inicial-note" class="text-muted"></small>
              </div>
            </div>
          </div>
          <div class="col-12 col-md-4">
            <div class="card h-100">
              <div class="card-body">
                <div class="small text-muted">Movimientos manuales (sesión)</div>
                <div class="d-flex align-items-center gap-2">
                  <span id="kpi-mov-count" class="badge bg-danger">0</span>
                  <span id="kpi-mov-total" class="fs-6 fw-semibold">—</span>
                </div>
                <small class="text-muted">Entradas – Salidas</small>
              </div>
            </div>
          </div>
          <div class="col-12 col-md-4">
            <div class="card h-100">
              <div class="card-body">
                <div class="small text-muted">Órdenes (cuentan a caja)</div>
                <div class="d-flex align-items-center gap-2">
                  <span id="kpi-ords-count" class="badge bg-danger">0</span>
                  <span id="kpi-ords-total" class="fs-6 fw-semibold">—</span>
                </div>
                <small class="text-muted">Estatus 11/12 (sin crédito)</small>
              </div>
            </div>
          </div>
        </div>

        <!-- Resumen -->
        <div id="caja-resumen" class="mt-1"></div>

        <!-- Tablas -->
        <div id="caja-ordenes"></div>
        <div id="caja-eliminadas"></div>
        <div id="caja-sin-tipo"></div>
        <div id="caja-movs"></div>

      </div>
    `;

    ensureModalManual();
    ensureModalCierre();
    ensureModalSolicitudes();
    ensureModalSolicitudesDetalle();
    ensureModalDiscrepanciasMes();
  }

  // ===== Modal: Movimiento manual =====
  function ensureModalManual() {
    if ($('#modal-manual-mov')) return;
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modal-manual-mov';
    modal.tabIndex = -1;
    modal.innerHTML = `
      <div class="modal-dialog modal-lg modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title">Movimiento manual</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
          </div>
          <div class="modal-body">
            <form id="form-manual-mov" class="row g-3">
              <div class="col-12 col-md-4">
                <label class="form-label">Tipo</label>
                <select id="mov-tipo" class="form-select">
                  <option value="ENTRADA">ENTRADA</option>
                  <option value="SALIDA">SALIDA</option>
                </select>
              </div>
              <div class="col-12 col-md-4">
                <label class="form-label">Origen</label>
                <select id="mov-origen" class="form-select">
                  <option value="MANUAL">MANUAL</option>
                  <option value="ORDEN">ORDEN</option>
                </select>
              </div>
              <div class="col-12 col-md-4" id="wrap-idorden" style="display:none;">
                <label class="form-label">IdOrden (si ORDEN)</label>
                <input id="mov-idorden" type="number" class="form-control" placeholder="Ej. 2488">
              </div>

              <div class="col-12">
                <label class="form-label">Concepto</label>
                <input id="mov-concepto" type="text" class="form-control" placeholder="Descripción del movimiento" />
              </div>

              <div class="col-12 col-md-4">
                <label class="form-label">Tipo de pago</label>
                <select id="mov-tipopago" class="form-select">
                  <option value="">SinTipodePago</option>
                  <option value="01">Efectivo</option>
                  <option value="03">Transferencia</option>
                  <option value="04">Tarjeta Crédito</option>
                  <option value="28">Tarjeta Débito</option>
                  <option value="02">Cheque</option>
                </select>
              </div>

              <div class="col-12 col-md-4" id="wrap-fact" style="display:none;">
                <label class="form-label">Comprobante</label>
                <select id="mov-fact" class="form-select">
                  <option value="12">Pago sin factura (Estatus 12)</option>
                  <option value="11">Pago con factura (Estatus 11)</option>
                  <option value="25">Facturado Pago contra Entrega (Estatus 25)</option>
                </select>
              </div>
              <div class="col-12 col-md-4" id="wrap-parcial" style="display:none;">
                <label class="form-label d-block">Pago parcial</label>
                <div class="form-check form-switch">
                  <input class="form-check-input" type="checkbox" id="mov-parcial">
                  <label class="form-check-label" for="mov-parcial">Marcar como parcial (informativo)</label>
                </div>
              </div>

              <div class="col-12 col-md-4">
                <label class="form-label">Monto</label>
                <input id="mov-monto" type="number" step="0.01" class="form-control" placeholder="0.00" />
              </div>
            </form>
          </div>
          <div class="modal-footer">
            <button id="btn-save-mov" type="button" class="btn btn-danger">Guardar</button>
            <button type="button" class="btn btn-outline-dark" data-bs-dismiss="modal">Cerrar</button>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(modal);

    $('#mov-origen').addEventListener('change', () => {
      const isOrden = $('#mov-origen').value === 'ORDEN';
      $('#wrap-idorden').style.display = isOrden ? '' : 'none';
      $('#wrap-fact').style.display    = isOrden ? '' : 'none';
      $('#wrap-parcial').style.display = isOrden ? '' : 'none';
    });
    $('#btn-save-mov').addEventListener('click', onSaveManualMov);
  }

  // ===== Modal: Solicitar cierre =====
  function ensureModalCierre() {
    if ($('#modal-solicitar-cierre')) return;
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modal-solicitar-cierre';
    modal.tabIndex = -1;
    modal.innerHTML = `
      <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title">Solicitar cierre de caja</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
          </div>
          <div class="modal-body">
            <div class="alert alert-info">
              Verifica los montos que tienes físicamente por cada forma de pago.<br/>
              Si no coinciden con lo esperado, podrás enviar igual (aparecerá advertencia).
            </div>

            <div class="row g-3" id="cierre-inputs"></div>

            <div class="mt-3">
              <label class="form-label">Observaciones (opcional)</label>
              <textarea id="cierre-observ" class="form-control" rows="2" placeholder="Notas para supervisor"></textarea>
            </div>
          </div>
          <div class="modal-footer">
            <button id="btn-send-cierre" type="button" class="btn btn-dark">Enviar solicitud</button>
            <button type="button" class="btn btn-outline-dark" data-bs-dismiss="modal">Cerrar</button>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(modal);
    $('#btn-send-cierre').addEventListener('click', onSendCierre);
  }

  // ===== Modal: Solicitudes =====
  function ensureModalSolicitudes() {
    if ($('#modal-solicitudes')) return;
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modal-solicitudes';
    modal.tabIndex = -1;
    modal.innerHTML = `
      <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title">Solicitudes de cierre</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
          </div>
          <div class="modal-body">
            <div class="mb-3">
              <span class="badge bg-danger me-2">Pendientes</span>
              <span class="badge bg-dark">Historial (3 días)</span>
            </div>
            <div id="solicitudes-pend"></div>
            <hr/>
            <div id="solicitudes-hist"></div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-outline-dark" data-bs-dismiss="modal">Cerrar</button>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(modal);
  }

  // ===== Modal: Detalle de solicitud =====
  function ensureModalSolicitudesDetalle() {
    if ($('#modal-sol-detalle')) return;
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modal-sol-detalle';
    modal.tabIndex = -1;
    modal.innerHTML = `
      <div class="modal-dialog modal-lg modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h6 class="modal-title">Detalle de montos</h6>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
          </div>
          <div class="modal-body" id="sol-detalle-body"></div>
          <div class="modal-footer">
            <button type="button" class="btn btn-outline-dark" data-bs-dismiss="modal">Cerrar</button>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(modal);
  }

  // ===== Modal: Discrepancias (mes) con loader y select de usuarios =====
  function ensureModalDiscrepanciasMes() {
    if ($('#modal-discrepancias-mes')) return;
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modal-discrepancias-mes';
    modal.tabIndex = -1;
    modal.innerHTML = `
      <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title">Discrepancias por rango de fechas</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
          </div>
          <div class="modal-body">
            <form id="form-disc-filtros" class="row g-2 mb-2">
              <div class="col-12 col-md-3">
                <label class="form-label">Fecha inicial</label>
                <input id="disc-fini" type="date" class="form-control" />
              </div>
              <div class="col-12 col-md-3">
                <label class="form-label">Fecha final</label>
                <input id="disc-ffin" type="date" class="form-control" />
              </div>
              <div class="col-12 col-md-4">
                <label class="form-label">Usuario</label>
                <select id="disc-user-sel" class="form-select">
                  <option value="0">Todos</option>
                </select>
                <input id="disc-user" type="number" class="form-control mt-2" placeholder="IdUsuario (fallback)" style="display:none;" />
              </div>
              <div class="col-12 col-md-2 d-flex align-items-end">
                <button id="btn-disc-buscar" type="button" class="btn btn-danger w-100">Buscar</button>
              </div>
            </form>

            <!-- Loader -->
            <div id="disc-loader" class="d-none py-4 text-center">
              <div class="spinner-border" role="status" aria-hidden="true"></div>
              <div class="small mt-2">Cargando discrepancias…</div>
            </div>

            <div class="table-responsive">
              <table class="table table-sm align-middle">
                <thead>
                  <tr>
                    <th>Fecha cierre</th>
                    <th>Sesión</th>
                    <th>Usuario</th>
                    <th>Tipo pago</th>
                    <th class="text-end">Esperado</th>
                    <th class="text-end">Contado</th>
                    <th class="text-end">Diferencia</th>
                  </tr>
                </thead>
                <tbody id="disc-tbody"></tbody>
              </table>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-outline-dark" data-bs-dismiss="modal">Cerrar</button>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(modal);

    // Sincroniza hidden
    $('#disc-user-sel').addEventListener('change', () => {
      $('#disc-user').value = $('#disc-user-sel').value || '0';
    });
    $('#btn-disc-buscar').addEventListener('click', loadDiscrepanciasMes);
  }

  // ===== Estado =====
  let state = {
    session: null,
    resumen: [],
    ordenes: [],
    eliminadas: [],
    sin_tipo: [],
    movs: [],
    prevday: null,
    lastDisc: null,
  };

  // ===== Renders =====
  function renderDiscrepanciaBanner() {
    const box = $('#caja-discrepancia-banner');
    if (!box) return;
    box.innerHTML = '';
    const d = state.lastDisc;
    if (!d || !d.cierre) return;

    const totalDif = Number(d.cierre.diferencia || 0);
    const items = (d.items || []).map(x =>
      `<span class="badge bg-danger me-1">${x.tipo_pago} ${fmt(x.diferencia)}</span>`
    ).join(' ');

    box.innerHTML = `
      <div class="alert alert-warning d-flex justify-content-between align-items-center">
        <div>
          <strong>Tu cierre anterior tuvo discrepancias.</strong>
          <span class="ms-2">Total: ${fmt(totalDif)}</span>
          ${items ? `<div class="mt-1">${items}</div>`:''}
        </div>
        <button type="button" class="btn btn-sm btn-outline-dark" data-bs-dismiss="alert" aria-label="Cerrar">Ocultar</button>
      </div>
    `;
  }

  function renderSessionBanner() {
    const b = $('#caja-session-banner');
    const hint = $('#caja-prevday-hint');
    if (!b) return;

    if (!state.session) {
      b.textContent = '—';
      if (hint) hint.innerHTML = '';
      return;
    }
    const s = state.session;
    b.innerHTML = `
      Sesión #<span class="badge bg-dark">${s.idSesion}</span>
      — <span class="text-muted">Apertura:</span> ${s.fecha_apertura || '-'}
      — <span class="text-muted">Estado:</span> <span class="badge ${s.estado==='ABIERTA'?'bg-success':'bg-dark'}">${s.estado}</span>
    `;

    if (hint) {
      if (state.prevday && state.prevday.length) {
        const p = state.prevday[0];
        hint.innerHTML =
          `<span class="badge bg-warning text-dark">Atención</span>
           Caja anterior sin cierre — Sesión #${p.idSesion}, día ${p.fecha_dia || '-'}.
           <span class="text-muted">Debes aprobar/rechazar el cierre pendiente.</span>`;
      } else {
        hint.innerHTML = '';
      }
    }
  }

  function renderKPIs() {
    const saldoIni = $('#kpi-saldo-inicial');
    const saldoNote = $('#kpi-saldo-inicial-note');
    if (saldoIni) saldoIni.textContent = fmt(state.session?.saldo_inicial || 0);
    if (saldoNote) saldoNote.textContent = state.session ? `IdUsuario ${state.session.IdUsuario} · Sesión ${state.session.idSesion}` : '';

    const cnt = $('#kpi-mov-count');
    const tot = $('#kpi-mov-total');
    if (cnt) cnt.textContent = state.movs.length;
    if (tot) {
      const entradas = state.movs.filter(m=>m.tipo==='ENTRADA').reduce((a,b)=>a+Number(b.monto||0),0);
      const salidas  = state.movs.filter(m=>m.tipo==='SALIDA' ).reduce((a,b)=>a+Number(b.monto||0),0);
      tot.textContent = `${fmt(entradas - salidas)} ( ${fmt(entradas)} − ${fmt(salidas)} )`;
    }

    const ordsCaja = state.ordenes.filter(o=>o.categoria==='CAJA');
    const cntO = $('#kpi-ords-count');
    const totO = $('#kpi-ords-total');
    if (cntO) cntO.textContent = ordsCaja.length;
    if (totO) {
      const sum = ordsCaja.reduce((a,b)=>a+Number(b.total||0),0);
      totO.textContent = fmt(sum);
    }
  }

  function renderResumenPagos() {
    const resumen = state.resumen || [];
    const box = $('#caja-resumen');
    if (!box) return;
    box.innerHTML = '';

    const CAJA = resumen.filter(r => r.categoria === 'CAJA');
    const PARCIAL = resumen.filter(r => r.categoria === 'PARCIAL');
    const CREDITO = resumen.filter(r => r.categoria === 'CREDITO');

    const makeTable = (title, rows, note='') => {
      const wrap = document.createElement('div');
      wrap.className = 'card mb-3';
      wrap.innerHTML = `
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-center mb-2">
            <h6 class="m-0">${title}</h6>
            ${note ? `<small class="text-muted">${note}</small>` : ''}
          </div>
          <div class="table-responsive">
            <table class="table align-middle table-sm">
              <thead>
                <tr>
                  <th>Tipo de pago</th>
                  <th class="text-end">Pagadas</th>
                  <th class="text-end">Entradas</th>
                  <th class="text-end">Salidas</th>
                  <th class="text-end">Total caja</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
        </div>`;
      const tbody = wrap.querySelector('tbody');
      rows.forEach(r=>{
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td><span class="badge bg-danger">${r.tipo_pago || 'SinTipodePago'}</span></td>
          <td class="text-end">${fmt(r.total_pagadas||0)}</td>
          <td class="text-end">${fmt(r.entradas_manuales||0)}</td>
          <td class="text-end">${fmt(r.salidas_manuales||0)}</td>
          <td class="text-end fw-semibold">${fmt(r.total_caja_hoy||0)}</td>`;
        tbody.appendChild(tr);
      });
      return wrap;
    };

    if (CAJA.length) box.appendChild(makeTable('Resumen por forma de pago — Caja (cuenta)', CAJA, 'Solo Estatus 11/12 + manuales'));
    if (PARCIAL.length) box.appendChild(makeTable('Pago Parcial (informativo)', PARCIAL, 'No suma a caja'));
    if (CREDITO.length) box.appendChild(makeTable('Crédito (informativo)', CREDITO, 'No suma a caja'));
  }

  function renderOrdenes() {
    const cont = $('#caja-ordenes'); if (!cont) return;
    cont.innerHTML = '';
    const make = (title, rows) => {
      const card = document.createElement('div');
      card.className = 'card mb-3';
      card.innerHTML = `
        <div class="card-body">
          <h6 class="mb-2">${title}</h6>
          <div class="table-responsive">
            <table class="table align-middle table-sm">
              <thead>
                <tr>
                  <th>Fecha</th>
                  <th>Orden</th>
                  <th>Cliente</th>
                  <th>Vendedor</th>
                  <th>Tipo pago</th>
                  <th>Estatus</th>
                  <th class="text-end">Total</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
        </div>`;
      const tbody = card.querySelector('tbody');
      rows.forEach(r=>{
        const url = `https://grupoxtinfire.com/admin/ordenes/detalle/${r.IdOrden}`;
        const badgeClass =
          r.categoria==='CREDITO' ? 'bg-warning text-dark'
          : r.categoria==='PARCIAL' ? 'bg-info text-dark'
          : 'bg-success';
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${r.Fecha || ''}</td>
          <td><a href="${url}" target="_blank">#${r.IdOrden}</a></td>
          <td>${r.cliente || ''}</td>
          <td>${r.vendedor || ''}</td>
          <td>${r.tipo_pago || 'SinTipodePago'}</td>
          <td><span class="badge ${badgeClass}">${r.estatus_text}</span></td>
          <td class="text-end">${fmt(r.total||0)}</td>`;
        tbody.appendChild(tr);
      });
      return card;
    };
    const ordCaja    = state.ordenes.filter(r => r.categoria === 'CAJA');
    const ordParcial = state.ordenes.filter(r => r.categoria === 'PARCIAL');
    const ordCred    = state.ordenes.filter(r => r.categoria === 'CREDITO');
    if (ordCaja.length)    cont.appendChild(make('Órdenes pagadas (cuentan a caja)', ordCaja));
    if (ordParcial.length) cont.appendChild(make('Órdenes con Pago Parcial (informativo)', ordParcial));
    if (ordCred.length)    cont.appendChild(make('Órdenes a Crédito (informativo)', ordCred));
  }

  function renderEliminadas() {
    const cont = $('#caja-eliminadas'); if (!cont) return;
    cont.innerHTML = '';
    const rows = state.eliminadas || [];
    if (!rows.length) return;
    const card = document.createElement('div');
    card.className = 'card mb-3';
    card.innerHTML = `
      <div class="card-body">
        <h6 class="mb-2">Órdenes eliminadas (referencia)</h6>
        <div class="table-responsive">
          <table class="table align-middle table-sm">
            <thead>
              <tr>
                <th>Fecha</th>
                <th>Orden</th>
                <th>Cliente</th>
                <th>Vendedor</th>
                <th>Tipo pago</th>
                <th>Estatus</th>
                <th class="text-end">Total ref.</th>
              </tr>
            </thead>
            <tbody></tbody>
          </table>
        </div>
      </div>`;
    const tbody = card.querySelector('tbody');
    rows.forEach(r=>{
      const url = `https://grupoxtinfire.com/admin/ordenes/detalle/${r.IdOrden}`;
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${r.Fecha || ''}</td>
        <td><a href="${url}" target="_blank">#${r.IdOrden}</a></td>
        <td>${r.cliente || ''}</td>
        <td>${r.vendedor || ''}</td>
        <td>${r.tipo_pago || 'SinTipodePago'}</td>
        <td><span class="badge bg-dark">Eliminada</span></td>
        <td class="text-end">${fmt(r.total_referencia||0)}</td>`;
      tbody.appendChild(tr);
    });
    cont.appendChild(card);
  }

  function renderSinTipo() {
    const cont = $('#caja-sin-tipo'); if (!cont) return;
    cont.innerHTML = '';
    const rows = state.sin_tipo || [];
    if (!rows.length) return;

    const card = document.createElement('div');
    card.className = 'card mb-3';
    card.innerHTML = `
      <div class="card-body">
        <h6 class="mb-2">Órdenes pagadas sin tipo de pago (corregir)</h6>
        <div class="table-responsive">
          <table class="table align-middle table-sm">
            <thead>
              <tr>
                <th>Fecha</th>
                <th>Orden</th>
                <th>Estatus</th>
                <th class="text-end">Total</th>
              </tr>
            </thead>
            <tbody></tbody>
          </table>
        </div>
      </div>`;
    const tbody = card.querySelector('tbody');
    rows.forEach(r=>{
      const url = `https://grupoxtinfire.com/admin/ordenes/detalle/${r.IdOrden}`;
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${r.Fecha || ''}</td>
        <td><a href="${url}" target="_blank">#${r.IdOrden}</a></td>
        <td><span class="badge bg-warning text-dark">${r.NombreEstatus || 'Pagada'}</span></td>
        <td class="text-end">${fmt(r.Total||0)}</td>`;
      tbody.appendChild(tr);
    });
    cont.appendChild(card);
  }

  function renderMovs() {
    const cont = $('#caja-movs'); if (!cont) return;
    cont.innerHTML = '';
    const rows = state.movs || [];
    const card = document.createElement('div');
    card.className = 'card mb-3';
    card.innerHTML = `
      <div class="card-body">
        <div class="d-flex align-items-center justify-content-between mb-2">
          <h6 class="m-0">Movimientos manuales (sesión)</h6>
          <span class="badge bg-danger">${rows.length}</span>
        </div>
        <div class="table-responsive">
          <table class="table align-middle table-sm">
            <thead>
              <tr>
                <th>Fecha</th>
                <th>Tipo</th>
                <th>Origen</th>
                <th>Orden</th>
                <th>Tipo pago</th>
                <th>Concepto</th>
                <th class="text-end">Monto</th>
                <th>Creado por</th>
              </tr>
            </thead>
            <tbody></tbody>
          </table>
        </div>
      </div>`;
    const tbody = card.querySelector('tbody');
    rows.forEach(m=>{
      const fecha = m.creado_en_fmt || m.creado_en || '';
      const ordenTxt = m.IdOrden ? `<a href="https://grupoxtinfire.com/admin/ordenes/detalle/${m.IdOrden}" target="_blank">#${m.IdOrden}</a>` : '—';
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>${fecha}</td>
        <td><span class="badge ${m.tipo==='ENTRADA'?'bg-success':'bg-danger'}">${m.tipo}</span></td>
        <td>${m.origen}</td>
        <td>${ordenTxt}</td>
        <td>${m.tipo_pago || m.TipoPago || 'SinTipodePago'}</td>
        <td>${m.concepto || ''}</td>
        <td class="text-end">${fmt(m.monto||0)}</td>
        <td><span class="badge ${theme()==='dark'?'bg-info text-dark':'bg-dark'}">${m.creado_por_nombre || m.creado_por || ''}</span></td>`;
      tbody.appendChild(tr);
    });
    cont.appendChild(card);
  }

  // ===== Cierre modal =====
  function openCierreModal() {
    if (!state.session) {
      Toast.fire({ icon: "error", title: "No hay sesión abierta", theme: colorswal });
      return;
    }
    const expected = computeExpectedTotals();
    const formas = [
      'Efectivo','Transferencia','Tarjeta Crédito','Tarjeta Débito','Cheque','SinTipodePago','Pago desconocido'
    ];
    const wrap = $('#cierre-inputs');
    wrap.innerHTML = '';
    formas.forEach(nombre=>{
      const esp = Number(expected[nombre] || 0);
      const id = 'fp_' + nombre.replace(/\s+/g,'_').toLowerCase();
      const col = document.createElement('div');
      col.className = 'col-12 col-md-4';
      col.innerHTML = `
        <label class="form-label d-flex justify-content-between align-items-center">
          <span>${nombre}</span>
          <span class="badge bg-danger">Debe tener: ${fmt(esp)}</span>
        </label>
        <input type="number" step="0.01" class="form-control" id="${id}" placeholder="0.00" />
      `;
      wrap.appendChild(col);
    });

    new bootstrap.Modal($('#modal-solicitar-cierre')).show();
  }

  async function onSendCierre() {
    try {
      if (!state.session) {
        Toast.fire({ icon: "error", title: "No hay sesión abierta", theme: colorswal });
        return;
      }
      // Leemos lo contado
      const readVal = (name) => {
        const id = '#fp_' + name.replace(/\s+/g,'_').toLowerCase();
        return Number($(id)?.value || 0);
      };
      const payload = {
        efectivo: readVal('Efectivo'),
        transferencia: readVal('Transferencia'),
        tarjeta_credito: readVal('Tarjeta Crédito'),
        tarjeta_debito: readVal('Tarjeta Débito'),
        cheque: readVal('Cheque'),
        sintipodepago: readVal('SinTipodePago'),
        desconocido: readVal('Pago desconocido'),
        observ: $('#cierre-observ')?.value || ''
      };

      // Esperado con el mismo criterio (por sesión)
      const expected = computeExpectedTotals();

      // Diffs
      const pairs = [
        ['Efectivo', 'efectivo'],
        ['Transferencia','transferencia'],
        ['Tarjeta Crédito','tarjeta_credito'],
        ['Tarjeta Débito','tarjeta_debito'],
        ['Cheque','cheque'],
        ['SinTipodePago','sintipodepago'],
        ['Pago desconocido','desconocido'],
      ];
      const diffs = [];
      pairs.forEach(([nom,key])=>{
        const exp = Number(expected[nom]||0);
        const got = Number(payload[key]||0);
        const delta = +(got - exp).toFixed(2);
        if (Math.abs(delta) > 0.009) diffs.push({ nom, exp, got, delta });
      });

      const proceed = async () => {
        const resumen_json = JSON.stringify({
          session: state.session?.idSesion,
          esperado: expected,
          contado: payload,
          observ: payload.observ
        });
        await apiPost('caja_solicitar_cierre', {
          idSesion: state.session.idSesion,
          IdUsuario: state.session.IdUsuario,
          resumen_json
        });
        Toast.fire({ icon: "success", title: "Solicitud enviada", theme: colorswal });
        await loadAll();
        bootstrap.Modal.getInstance($('#modal-solicitar-cierre'))?.hide();
      };

      if (diffs.length) {
        const html = diffs.map(d=>`<div><strong>${d.nom}:</strong> Esperado ${fmt(d.exp)} · Contado ${fmt(d.got)} · Δ ${fmt(d.delta)}</div>`).join('');
        const r = await Swal.fire({
          icon: 'warning',
          title: 'Diferencias detectadas',
          html: `${html}<hr class="my-2"/><div class="text-muted">¿Enviar de todos modos?</div>`,
          showCancelButton: true,
          confirmButtonText: 'Enviar igual',
          cancelButtonText: 'Revisar',
        });
        if (!r.isConfirmed) return;
      }
      await proceed();

    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    }
  }

  // ===== Solicitudes =====
  async function openSolicitudesModal() {
    try {
      const contP = $('#solicitudes-pend');
      const contH = $('#solicitudes-hist');
      contP.innerHTML = '<div class="alert alert-info m-0">Cargando…</div>';
      contH.innerHTML = '<div class="alert alert-info m-0">Cargando…</div>';

      const [pend, hist] = await Promise.all([
        apiPost('caja_listar_solicitudes', {}),
        apiPost('caja_listar_solicitudes_hist', {})
      ]);

      renderSolicitudesList(contP, pend || [], true);
      renderSolicitudesList(contH, hist || [], false);

      new bootstrap.Modal($('#modal-solicitudes')).show();
    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    }
  }

  function renderSolicitudesList(container, list, isPendientes) {
    container.innerHTML = '';
    if (!list || !list.length) {
      container.innerHTML = `<div class="alert alert-info m-0">Sin registros.</div>`;
      return;
    }

    const depUser = Number($('#depparanoti')?.value || 0);
    const isSupervisor = isSupervisorDept(depUser);

    list.forEach(s=>{
      let resumen = {};
      try { resumen = JSON.parse(s.resumen_json || '{}'); } catch {}
      const estado = s.estado;

      const headerBadges = `
        <span class="badge ${estado==='PENDIENTE'?'bg-warning text-dark' : estado==='APROBADA'?'bg-success':'bg-danger'}">${estado}</span>
        <span class="badge bg-dark">Solicitud #${s.idSolicitud}</span>
        <span class="badge bg-danger">Sesión #${s.idSesion}</span>
        <span class="badge ${theme()==='dark'?'bg-info text-dark':'bg-dark'}">Usuario: ${s.Usuario || s.IdUsuario}</span>
        <span class="badge bg-info text-dark">Depto: ${s.Departamento}</span>
      `;

      const showObsRechazo = (estado==='RECHAZADA' && s.obs_supervisor);
      const footBtns = [];
      if (isSupervisor && estado==='PENDIENTE') {
        footBtns.push(`<button class="btn btn-sm btn-success" data-approve="${s.idSolicitud}" data-sid="${s.idSesion}">Aprobar</button>`);
        footBtns.push(`<button class="btn btn-sm btn-outline-danger" data-reject="${s.idSolicitud}">Rechazar</button>`);
      }
      footBtns.push(`<button class="btn btn-sm btn-outline-dark" data-detalle='${JSON.stringify(resumen).replaceAll("'", "&apos;")}'>Detalle</button>`);

      const card = document.createElement('div');
      card.className = 'card mb-3';
      card.innerHTML = `
        <div class="card-body">
          <div class="d-flex justify-content-between align-items-start">
            <div>
              <div class="mb-1">${headerBadges}</div>
              <div class="small text-muted mb-2">
                Solicitada: ${s.solicitado_en || '-'}
                ${s.resuelto_en ? ' · Resuelta: '+s.resuelto_en : ''}
                ${showObsRechazo ? ' · <span class="text-danger">Motivo rechazo: '+s.obs_supervisor+'</span>' : ''}
              </div>
              <div class="small text-muted">Apertura: ${s.fecha_apertura} · Saldo inicial: ${fmt(s.saldo_inicial||0)}</div>
            </div>
            <div class="d-flex flex-column gap-2">
              ${footBtns.join('')}
            </div>
          </div>
        </div>
      `;
      container.appendChild(card);
    });

    container.onclick = async (ev) => {
      const btnA = ev.target.closest('[data-approve]');
      const btnR = ev.target.closest('[data-reject]');
      const btnD = ev.target.closest('[data-detalle]');

      if (btnD) {
        const resumen = JSON.parse(btnD.getAttribute('data-detalle') || '{}');
        const esperado = resumen.esperado || {};
        const contado  = resumen.contado || {};
        const html = `
          <div class="row g-2 small">
            <div class="col-12 col-md-4"><strong>Efectivo</strong>: esp. ${fmt(esperado['Efectivo']||0)} · cnt. ${fmt(contado.efectivo||0)}</div>
            <div class="col-12 col-md-4"><strong>Transferencia</strong>: esp. ${fmt(esperado['Transferencia']||0)} · cnt. ${fmt(contado.transferencia||0)}</div>
            <div class="col-12 col-md-4"><strong>Tarj. Crédito</strong>: esp. ${fmt(esperado['Tarjeta Crédito']||0)} · cnt. ${fmt(contado.tarjeta_credito||0)}</div>
            <div class="col-12 col-md-4"><strong>Tarj. Débito</strong>: esp. ${fmt(esperado['Tarjeta Débito']||0)} · cnt. ${fmt(contado.tarjeta_debito||0)}</div>
            <div class="col-12 col-md-4"><strong>Cheque</strong>: esp. ${fmt(esperado['Cheque']||0)} · cnt. ${fmt(contado.cheque||0)}</div>
            <div class="col-12 col-md-4"><strong>SinTipo</strong>: esp. ${fmt(esperado['SinTipodePago']||0)} · cnt. ${fmt(contado.sintipodepago||0)}</div>
            ${resumen.observ?`<div class="col-12 text-muted">Obs: ${resumen.observ}</div>`:''}
          </div>`;
        $('#sol-detalle-body').innerHTML = html;
        new bootstrap.Modal($('#modal-sol-detalle')).show();
      }

      if (btnA) {
        const idSolicitud = btnA.getAttribute('data-approve');
        const idSesion = btnA.getAttribute('data-sid');
        try {
          const IdUsuario = Number($('#Idusu')?.value || 0);
          await apiPost('caja_aprobar_cierre', { idSolicitud, idSesion, resuelto_por: IdUsuario });
          Toast.fire({ icon: "success", title: "Cierre aprobado", theme: colorswal });

          if (idSesion) window.open(PDF_CIERRE_URL + encodeURIComponent(idSesion), '_blank');

          await loadAll();
          await openSolicitudesModal();
        } catch (e) {
          Toast.fire({ icon: "error", title: e.message, theme: colorswal });
        }
      }

      if (btnR) {
        const idSolicitud = btnR.getAttribute('data-reject');
        const { value: obs } = await Swal.fire({
          title: 'Motivo del rechazo',
          input: 'text',
          inputPlaceholder: 'Escribe la razón…',
          showCancelButton: true,
          confirmButtonText: 'Rechazar',
          cancelButtonText: 'Cancelar'
        });
        if (obs === undefined) return;
        try {
          const IdUsuario = Number($('#Idusu')?.value || 0);
          await apiPost('caja_rechazar_cierre', { idSolicitud, resuelto_por: IdUsuario, obs });
          Toast.fire({ icon: "success", title: "Solicitud rechazada", theme: colorswal });
          await loadAll();
          await openSolicitudesModal();
        } catch (e) {
          Toast.fire({ icon: "error", title: e.message, theme: colorswal });
        }
      }
    };
  }

  // ===== Discrepancias (mes) =====
  async function loadUsuariosSelect() {
    try {
      const sel = $('#disc-user-sel');
      const inputFallback = $('#disc-user');
      sel.innerHTML = `<option value="0">Todos</option>`;
      inputFallback.style.display = 'none';

      const rows = await apiPost('caja_usuarios_list', {}); // [{IdUsuario, Usuario}]
      if (!rows || !rows.length) {
        inputFallback.style.display = '';
        return;
      }
      rows.forEach(u=>{
        const opt = document.createElement('option');
        opt.value = String(u.IdUsuario);
        opt.textContent = `${u.Usuario || ('Usuario '+u.IdUsuario)} (#${u.IdUsuario})`;
        sel.appendChild(opt);
      });
      $('#disc-user').value = sel.value || '0';
    } catch (e) {
      $('#disc-user').style.display = '';
    }
  }

  async function loadDiscrepanciasMes() {
    try {
      // Mostrar loader
      $('#disc-loader')?.classList.remove('d-none');
      const tbody = $('#disc-tbody');
      if (tbody) tbody.innerHTML = '';

      const now = new Date();
      const first = new Date(now.getFullYear(), now.getMonth(), 1).toISOString().slice(0,10);
      const today = new Date().toISOString().slice(0,10);

      const fini = $('#disc-fini')?.value || first;
      const ffin = $('#disc-ffin')?.value || today;
      const user = Number(($('#disc-user-sel')?.value || $('#disc-user')?.value || 0));

      const rows = await apiPost('caja_discrepancias_list', { fini, ffin, IdUsuario: user });

      if (!tbody) return;
      tbody.innerHTML = '';
      if (!rows || !rows.length) {
        tbody.innerHTML = `<tr><td colspan="7" class="text-center text-muted">Sin discrepancias en el rango.</td></tr>`;
        return;
      }
      rows.forEach(r=>{
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${r.fecha_cierre}</td>
          <td>#${r.idSesion}</td>
          <td>${r.Usuario || r.IdUsuario}</td>
          <td>${r.tipo_pago}</td>
          <td class="text-end">${fmt(r.esperado||0)}</td>
          <td class="text-end">${fmt(r.contado||0)}</td>
          <td class="text-end">${fmt(r.diferencia||0)}</td>`;
        tbody.appendChild(tr);
      });
    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    } finally {
      // Ocultar loader
      $('#disc-loader')?.classList.add('d-none');
    }
  }

  // ===== Eventos =====
  async function onOpenCaja() {
    try {
      const IdUsuario = Number($('#Idusu')?.value || 0);
      const Departamento = Number($('#depparanoti')?.value || 0);
      const { dia } = todayRange();

      const { value: monto } = await Swal.fire({
        title: 'Saldo inicial',
        input: 'number',
        inputAttributes: { step: '0.01' },
        inputValue: 0,
        showCancelButton: true,
        confirmButtonText: 'Abrir caja',
        cancelButtonText: 'Cancelar'
      });
      if (monto === undefined) return;

      await apiPost('caja_open', {
        IdUsuario,
        Departamento,
        saldo_inicial: Number(monto||0).toFixed(2),
        fecha_dia: dia,
        dia
      });
      Toast.fire({ icon: "success", title: "Caja abierta", theme: colorswal });
      await loadAll();
    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    }
  }

  async function onSaveManualMov() {
    try {
      if (!state.session) {
        Toast.fire({ icon: "error", title: "No hay sesión abierta", theme: colorswal });
        return;
      }
      const idSesion = state.session.idSesion;
      const tipo = $('#mov-tipo').value;
      const origen = $('#mov-origen').value;
      const IdOrden = Number($('#mov-idorden')?.value || 0) || null;
      const concepto = $('#mov-concepto').value || '';
      const TipoPago = $('#mov-tipopago').value || null;
      const estatus  = $('#mov-fact') ? $('#mov-fact').value : '';
      const monto = Number($('#mov-monto')?.value || 0);

      if (!tipo || !origen || !monto || !concepto) {
        Toast.fire({ icon: "error", title: "Completa tipo, origen, concepto y monto", theme: colorswal });
        return;
      }
      if (origen === 'ORDEN' && !IdOrden) {
        Toast.fire({ icon: "error", title: "IdOrden requerido si el origen es ORDEN", theme: colorswal });
        return;
      }

      await apiPost('caja_add_mov', {
        idSesion,
        tipo,
        origen,
        IdOrden: IdOrden || '',
        IdDetalle: '',
        concepto,
        TipoPago: TipoPago || '',
        estatus: estatus || '',
        monto: Number(monto||0).toFixed(2),
        creado_por: Number($('#Idusu')?.value || 0)
      });

      if (origen === 'ORDEN') {
        Toast.fire({ icon: "success", title: "Movimiento registrado. Se abrirá la orden.", theme: colorswal });
        window.open(`https://grupoxtinfire.com/admin/ordenes/detalle/${IdOrden}`, '_blank');
      } else {
        Toast.fire({ icon: "success", title: "Movimiento registrado", theme: colorswal });
      }

      await loadAll();
      bootstrap.Modal.getInstance($('#modal-manual-mov'))?.hide();
    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    }
  }

  async function onReimprimirTicket() {
    try {
      const IdUsuario = Number($('#Idusu')?.value || 0);
      console.log(IdUsuario)
      // 1) Intentar obtener último cierre
      try {
        const last = await apiPost('caja_last_cierre', { IdUsuario }); // { idSesion }
        console.log(last)
        if (last && last.idSesion) {
          window.open(PDF_CIERRE_URL + encodeURIComponent(last.idSesion), '_blank');
          return;
        }
      } catch(_){ /* fallback abajo */ }

      
    } catch (e) {
      Toast.fire({ icon: "error", title: e.message, theme: colorswal });
    }
  }

  // ===== Cargas =====
  async function loadAll() {
    try {
      const root = $('#'+ROOT_ID);
      if (!root) return;

      const IdUsuario = Number($('#Idusu')?.value || 0);
      const Departamento = Number($('#depparanoti')?.value || 0);
      const isSupervisor = isSupervisorDept(Departamento);
      $('#btn-discrepancias-mes').style.display = isSupervisor ? '' : 'none';

      const { ini, fin } = todayRange();

      // Sesión del día + previas sin cerrar
      const [sessionArr, prevday] = await Promise.all([
        apiPost('caja_session_today', { IdUsuario, Departamento }),
        apiPost('caja_check_prevday', { IdUsuario, Departamento })
      ]);

      // Si NO hay sesión hoy pero SÍ hay una ABIERTA previa, usamos esa (bloquea abrir caja nueva)
      let session = (sessionArr && sessionArr[0]) || null;
      const prevOpen = (prevday || []).find(s => s.estado === 'ABIERTA');
      if (!session && prevOpen) session = prevOpen;

      state.session = session;
      state.prevday = prevday || [];

      // Control de botón "Abrir caja"
      const btnOpen = $('#btn-open-caja');
      if (btnOpen) btnOpen.disabled = !!(prevOpen && !sessionArr?.length);

      renderSessionBanner();

      // Última discrepancia (usuario)
      state.lastDisc = await apiPost('caja_last_discrepancia', { IdUsuario });
      renderDiscrepanciaBanner();

      if (!state.session) {
        const idList = ['kpi-saldo-inicial','kpi-mov-count','kpi-mov-total','kpi-ords-count','kpi-ords-total'];
        idList.forEach(id => { const el = $('#'+id); if (el) el.textContent = (id==='kpi-mov-count' || id==='kpi-ords-count') ? '0' : '—'; });
        ['caja-resumen','caja-ordenes','caja-eliminadas','caja-sin-tipo','caja-movs'].forEach(id => { const el = $('#'+id); if (el) el.innerHTML=''; });
        return;
      }

      // ⬇️ FILTRADO POR SESIÓN ACTIVA EN TODAS LAS CONSULTAS DE HOY
      const idSesion = state.session.idSesion;

      const [resumen, ordenes, eliminadas, sin_tipo] = await Promise.all([
        apiPost('caja_resumen_pagos_hoy', { IdUsuario, Departamento, idSesion, ini, fin }),
        apiPost('caja_ordenes_hoy',       { IdUsuario, Departamento, idSesion, ini, fin }),
        apiPost('caja_eliminadas_hoy',    { IdUsuario, Departamento, idSesion, ini, fin }),
        apiPost('caja_sin_tipo_hoy',      { IdUsuario, Departamento, idSesion, ini, fin })
      ]);
      state.resumen    = resumen || [];
      state.ordenes    = ordenes || [];
      state.eliminadas = eliminadas || [];
      state.sin_tipo   = sin_tipo || [];

      const movs = await apiPost('caja_movs_hoy', { idSesion });
      state.movs = movs || [];

      renderKPIs();
      renderResumenPagos();
      renderOrdenes();
      renderEliminadas();
      renderSinTipo();
      renderMovs();

    } catch (e) {
      Toast.fire({ icon: "error", title: e.message || "Error al cargar caja", theme: colorswal });
    }
  }

  // ===== Limpieza de backdrops huérfanos =====
  document.addEventListener('hidden.bs.modal', () => {
    $$('.modal-backdrop').forEach(b => b.remove());
    document.body.classList.remove('modal-open');
    document.body.style.overflow = '';
    document.body.style.paddingRight = '';
  });

  // ===== Bootstrap / Listeners =====
  document.addEventListener('DOMContentLoaded', async () => {
    const root = $('#'+ROOT_ID);
    if (!root) return;

    buildUI(root);

    // Botones
    $('#btn-open-caja').addEventListener('click', onOpenCaja);
    $('#btn-manual-mov').addEventListener('click', ()=> new bootstrap.Modal($('#modal-manual-mov')).show());
    $('#btn-solicitar-cierre').addEventListener('click', openCierreModal);
    $('#btn-ver-solicitudes').addEventListener('click', openSolicitudesModal);
    $('#btn-reimprimir-ticket').addEventListener('click', onReimprimirTicket);

    $('#btn-discrepancias-mes').addEventListener('click', async ()=>{
      // defaults rango
      const now = new Date();
      $('#disc-fini').value = new Date(now.getFullYear(), now.getMonth(), 1).toISOString().slice(0,10);
      $('#disc-ffin').value = new Date().toISOString().slice(0,10);
      $('#disc-user').value = '0';

      // abrir modal con loader visible
      $('#disc-loader')?.classList.remove('d-none');
      $('#disc-tbody').innerHTML = '';
      const m = new bootstrap.Modal($('#modal-discrepancias-mes'));
      m.show();

      // cargar datos
      try {
        await loadUsuariosSelect();
        await loadDiscrepanciasMes();
      } catch(e) {
        Toast.fire({ icon: "error", title: e.message, theme: colorswal });
      } finally {
        $('#disc-loader')?.classList.add('d-none');
      }
    });

    await loadAll();
  });

})();
