// /assets/js/camera.js
(() => {
  const $ = (s) => document.querySelector(s);

  const modal    = $('#userListModal');
  const stage    = modal?.querySelector('.cam-stage');
  const video    = $('#preview');
  const img      = $('#snapshot');
  const canvas   = $('#camCanvas');
  const statusEl = $('#camStatus');

  const btnShot    = $('#liShot');     // Tomar foto
  const btnUpload  = $('#liUpload');   // Subir
  const btnRetake  = $('#liRetake');   // Repetir
  const btnStop    = $('#liStop');     // Cerrar cámara
  const btnSwitch  = $('#liSwitch');   // Cambiar cámara

  let stream = null;
  let cameras = [];
  let currentDeviceId = null;
  let lastBlob = null;

  /* =========================
     Helpers UI
     ========================= */
  function setStatus(msg) {
    if (statusEl) statusEl.textContent = msg || '';
    layoutStage(); // reacomoda si cambia alto del status
  }

  function forceLayers() {
    if (stage) {
      stage.style.position = 'relative';
      stage.style.overflow = 'hidden';
      stage.style.background = '#000';
    }
    if (video) {
      video.style.position = 'absolute';
      video.style.inset    = '0';
      video.style.width    = '100%';
      video.style.height   = '100%';
      video.style.objectFit = 'cover';
      video.style.transform = 'scaleX(-1)'; // espejo SOLO en preview
      video.style.zIndex    = '1';
      video.setAttribute('playsinline', '');
      video.muted = true;
    }
    if (img) {
      img.style.position = 'absolute';
      img.style.inset    = '0';
      img.style.width    = '100%';
      img.style.height   = '100%';
      img.style.objectFit = 'cover';
      img.style.zIndex    = '2';
    }
  }

  // Mini refuerzo: al abrir/limpiar, preview visible y snapshot oculto
  function enforceDefaultVisibility() {
    if (img) {
      img.style.display = 'none';
      img.removeAttribute('src');
    }
    if (video) {
      video.style.display = 'block';
    }
    btnRetake && btnRetake.classList.add('d-none');
    btnShot   && btnShot.classList.remove('disabled');
    if (btnShot) { btnShot.style.opacity = ''; btnShot.style.pointerEvents = ''; }
  }

  function showPreviewUI() {
    if (video) video.style.display = 'block';
    if (img)   img.style.display   = 'none';
    btnRetake && btnRetake.classList.add('d-none');
    btnShot   && btnShot.classList.remove('disabled');
    if (btnShot) { btnShot.style.opacity = ''; btnShot.style.pointerEvents = ''; }
  }

  function showSnapshotUI() {
    if (img)   img.style.display   = 'block';
    if (video) video.style.display = 'none';
    btnRetake && btnRetake.classList.remove('d-none');
    btnShot   && btnShot.classList.add('disabled');
    if (btnShot) { btnShot.style.opacity = '.5'; btnShot.style.pointerEvents = 'none'; }
  }

  // Calcula altura disponible del stage (entre header/footer/status)
  function layoutStage() {
    if (!modal || !stage) return;
    const content = modal.querySelector('.modal-content');
    const header  = modal.querySelector('.modal-header');
    const footer  = modal.querySelector('.card-footer');

    const totalH  = content?.clientHeight || window.innerHeight;
    const headerH = header?.offsetHeight  || 0;
    const footerH = footer?.offsetHeight  || 0;
    const statusH = statusEl?.offsetHeight|| 0;

    const stageH = Math.max(240, totalH - headerH - footerH - statusH - 4);
    stage.style.height = stageH + 'px';
    forceLayers();
  }

  /* ====== 100vh REAL en móviles (safe-area) ====== */
  function setVhVar() {
    const vh = window.innerHeight * 0.01; // 1% del alto visible
    document.documentElement.style.setProperty('--vh', `${vh}px`);
  }
  let _vhResizeHandler = null;
  let _vhOrientHandler = null;
  function attachVhHandlers() {
    setVhVar();
    _vhResizeHandler = () => setVhVar();
    _vhOrientHandler = () => setTimeout(setVhVar, 150);
    window.addEventListener('resize', _vhResizeHandler, { passive: true });
    window.addEventListener('orientationchange', _vhOrientHandler, { passive: true });
  }
  function detachVhHandlers() {
    if (_vhResizeHandler) window.removeEventListener('resize', _vhResizeHandler);
    if (_vhOrientHandler) window.removeEventListener('orientationchange', _vhOrientHandler);
    _vhResizeHandler = _vhOrientHandler = null;
  }

  /* ====== Resize/orientación para layout del stage ====== */
  let onResize = null, onOrient = null;
  function attachResizeHandlers() {
    onResize = () => layoutStage();
    onOrient = () => setTimeout(layoutStage, 150);
    window.addEventListener('resize', onResize, { passive: true });
    window.addEventListener('orientationchange', onOrient, { passive: true });
  }
  function detachResizeHandlers() {
    if (onResize)  window.removeEventListener('resize', onResize);
    if (onOrient)  window.removeEventListener('orientationchange', onOrient);
    onResize = onOrient = null;
  }

  /* =========================
     Cámara
     ========================= */
  async function listCameras() {
    const devices = await navigator.mediaDevices.enumerateDevices();
    return devices.filter(d => d.kind === 'videoinput');
  }

  async function startByFacing(facingMode = 'user') {
    try {
      return await navigator.mediaDevices.getUserMedia({
        video: { facingMode: { exact: facingMode }, width:{ideal:1280}, height:{ideal:720} },
        audio: false
      });
    } catch {
      return navigator.mediaDevices.getUserMedia({
        video: { facingMode, width:{ideal:1280}, height:{ideal:720} },
        audio: false
      });
    }
  }

  async function startByDevice(deviceId) {
    return navigator.mediaDevices.getUserMedia({
      video: { deviceId: { exact: deviceId }, width:{ideal:1280}, height:{ideal:720} },
      audio: false
    });
  }

  function stopCamera() {
    try { if (stream) stream.getTracks().forEach(t => t.stop()); } catch {}
    stream = null;
    if (video) video.srcObject = null;
    setStatus('Cámara detenida.');
  }

  async function openCamera() {
    if (!navigator.mediaDevices?.getUserMedia) {
      setStatus('Tu navegador no soporta cámara.');
      return;
    }
    try {
      setStatus('Abriendo cámara...');
      stopCamera();
      layoutStage();
      enforceDefaultVisibility();

      stream = await startByFacing('user'); // frontal
      if (!video) throw new Error('Falta <video id="preview">');
      video.srcObject = stream;

      await video.play().catch(()=>{});
      video.addEventListener('loadedmetadata', layoutStage, { once: true });

      cameras = await listCameras();
      const frontIdx = cameras.findIndex(c => /front|frontal|user/i.test(c.label));
      currentDeviceId = frontIdx >= 0 ? cameras[frontIdx].deviceId : (cameras[0]?.deviceId || null);
      if (btnSwitch) btnSwitch.style.pointerEvents = cameras.length > 1 ? 'auto' : 'none';

      showPreviewUI();
      setStatus('Cámara lista.');
    } catch (e) {
      console.error('[camera] openCamera error:', e);
      setStatus('No se pudo abrir la cámara. Revisa HTTPS y permisos.');
    }
  }

  async function switchCamera(e) {
    e?.preventDefault?.();
    if (!cameras.length || !currentDeviceId) return;
    try {
      const idx = cameras.findIndex(c => c.deviceId === currentDeviceId);
      const nextId = cameras[(idx + 1) % cameras.length].deviceId;

      const newStream = await startByDevice(nextId);
      const old = stream;
      stream = newStream;
      video.srcObject = stream;
      await video.play().catch(()=>{});
      old?.getTracks()?.forEach(t => t.stop());

      currentDeviceId = nextId;
      showPreviewUI();
      setStatus('Cámara cambiada.');
    } catch (e2) {
      console.error('[camera] switchCamera error:', e2);
      setStatus('No se pudo cambiar de cámara.');
    }
  }

  // Captura tipo "cover" al tamaño visible del stage y genera preview (dataURL)
  function dataURLtoBlob(dataURL) {
    const arr = dataURL.split(',');
    const mime = arr[0].match(/:(.*?);/)[1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8 = new Uint8Array(n);
    while (n--) u8[n] = bstr.charCodeAt(n);
    return new Blob([u8], { type: mime });
  }

  async function takePhoto(e) {
    e?.preventDefault?.();
    if (!stream) { alert('Abre la cámara primero.'); return; }
    if (!canvas)  { alert('No hay canvas para capturar.'); return; }

    const sw = stage.clientWidth;
    const sh = stage.clientHeight;

    let vw = video.videoWidth;
    let vh = video.videoHeight;
    if (!vw || !vh) {
      await new Promise(r => setTimeout(r, 60));
      vw = video.videoWidth  || 1280;
      vh = video.videoHeight || 720;
    }

    canvas.width  = sw;
    canvas.height = sh;

    const ctx = canvas.getContext('2d');
    // Escala "cover": llena sin deformar (NO espejo al guardar)
    const vRatio = vw / vh;
    const sRatio = sw / sh;
    let dw, dh, dx, dy;
    if (sRatio > vRatio) {
      dw = sw; dh = sw / vRatio; dx = 0; dy = (sh - dh) / 2;
    } else {
      dw = sh * vRatio; dh = sh; dx = (sw - dw) / 2; dy = 0;
    }
    ctx.clearRect(0, 0, sw, sh);
    ctx.drawImage(video, dx, dy, dw, dh);

    const dataURL = canvas.toDataURL('image/jpeg', 0.92); // preview inmediato
    if (img) { img.src = dataURL; }
    lastBlob = dataURLtoBlob(dataURL);

    showSnapshotUI();
    setStatus('Foto lista para subir.');
  }

  function retakePhoto(e) {
    e?.preventDefault?.();
    showPreviewUI();
    lastBlob = null; // obliga a tomar nueva antes de subir
    setStatus('Vuelve a encuadrar y dispara.');
  }

  async function upload(e) {
    e?.preventDefault?.();
    if (!lastBlob) { alert('Toma una foto primero.'); return; }
    try {
      setStatus('Subiendo...');
      if (typeof window.uploadPhoto !== 'function') throw new Error('Falta upload.js');
      const res = await window.uploadPhoto(lastBlob, 'foto.jpg', '/subir.php');
      console.log('Servidor:', res);
      setStatus('Subida completada.');
      alert('Foto subida correctamente.');
    } catch (err) {
      console.error('[camera] upload error:', err);
      setStatus('Error al subir.');
      alert('Error al subir la foto: ' + err.message);
    }
  }

  /* =========================
     Ciclo de vida del modal
     ========================= */
  function onShown() {
    attachVhHandlers();       // 100vh real en móvil (safe-area)
    attachResizeHandlers();   // recalcula altura del stage
    layoutStage();
    forceLayers();
    enforceDefaultVisibility();
    openCamera();

    // Refuerzo por si play() se difiere
    setTimeout(() => { try { video && video.play && video.play(); } catch {} }, 60);
  }

  function onHidden() {
    stopCamera();
    detachResizeHandlers();
    detachVhHandlers();
    enforceDefaultVisibility();
    lastBlob = null;
  }

  /* =========================
     Eventos
     ========================= */
  btnShot    && btnShot.addEventListener('click', takePhoto);
  btnRetake  && btnRetake.addEventListener('click', retakePhoto);
  btnUpload  && btnUpload.addEventListener('click', upload);
  btnStop    && btnStop.addEventListener('click', (e) => { e.preventDefault(); onHidden(); });
  btnSwitch  && btnSwitch.addEventListener('click', switchCamera);

  if (modal) {
    modal.addEventListener('shown.bs.modal', onShown);
    modal.addEventListener('hidden.bs.modal', onHidden);
  }

  setStatus('Consejo: usa HTTPS en móviles para que funcione la cámara.');
})();
