(() => {
  "use strict";

  // ================== CONFIG ==================
  const BASE_URL   = "https://grupoxtinfire.com/admin";
  const ENDPOINTS  = {
    buscarCliente: `${BASE_URL}/ajax/codi_get_one_cliente`,
    buscarOrdenes: `${BASE_URL}/ajax/codi_get_one_orden`,
    buscarEventos: `${BASE_URL}/ajax/codi_get_eventos2`,
  };

  // Globales obligatorios (ajusta si aplica)
  const GLOBALS = { hook: "codi_hook", action: "load" };

  const INPUT_ID    = "buscar";
  const RECENTS_ID  = "busquedasrecientes";
  const STORAGE_KEY = "xtinfire_busquedas";
  const MAX_RECENTS = 20;

  // Palabras clave válidas
  const KEYWORDS = Array.from(new Set([
    "orden","ordenes","cotizaciones","cotizacion","factura","reporte",
    "curso","cursos","capacitacion","capacitaciones",
    "vale","fecha","traer","informacion","datos",
    "rfc","cliente","clientes","id","id cliente"
  ]));

  // Rutas para redirección
  const ROUTES = {
    orden:        "ordenes/detalle",
    ordenes:      "ordenes/detalle",
    cotizacion:   "cotizaciones/detalle",
    cotizaciones: "cotizaciones/detalle",
  };

  // Handlers dinámicos (extensible)
  const API_HANDLERS = {
    // Cursos/Capacitaciones
    curso:          handlerCapacitaciones,
    cursos:         handlerCapacitaciones,
    capacitacion:   handlerCapacitaciones,
    capacitaciones: handlerCapacitaciones,
  };

  // ================== INIT ==================
  document.addEventListener("DOMContentLoaded", () => {
    const input = document.getElementById(INPUT_ID);
    if (!input) return;

    renderRecents();

    // Detectar Enter con foco en el input
    input.addEventListener("keydown", (e) => {
      if (e.key === "Enter") {
        e.preventDefault();
        alert("Detecte el input")
        processQuery(input.value);
      }
    });

    // Click en búsquedas recientes
    const cont = document.getElementById(RECENTS_ID);
    if (cont) {
      cont.addEventListener("click", (e) => {
        const a = e.target.closest("a.kewords");
        if (!a) return;
        const text = a.querySelector("span")?.textContent?.trim();
        if (text) {
          input.value = text;
          processQuery(text);
        }
      });
    }

    // Botón para borrar TODAS las búsquedas recientes (con confirmación + Toast)
    const btnBorrar = document.getElementById("borrarbusqueda");
    if (btnBorrar) {
      btnBorrar.addEventListener("click", () => {
        confirmClearSearches();
      });
    }
  });

  // ================== CORE ==================
  async function processQuery(raw) {
    const original = (raw || "").trim();
    if (!original) return;

    const qNorm = normalize(original);
    const tokens = qNorm.split(/\s+/);
    const kw = detectarKeyword(tokens);

    if (!kw) {
      saveSearch(original);
      renderRecents();
      return;
    }

    // Redirección directa: "orden 5171" / "cotizacion 5555" → pestaña nueva
    if ((kw === "orden" || kw === "ordenes" || kw === "cotizacion" || kw === "cotizaciones")) {
      const idDirecto = extractId(qNorm);
      if (idDirecto) {
        const route = ROUTES[kw];
        if (route) {
          const url = `${BASE_URL}/${route}/${encodeURIComponent(idDirecto)}`;
          window.open(url, "_blank");
          saveSearch(original);
          renderRecents();
          return;
        }
      }
      // Si no hay número, continuará al flujo cliente->órdenes
    }

    // "id cliente 123" → ir directo por órdenes/cotizaciones
    const idCli = extractIdCliente(qNorm);
    const quiereCotizacion = tokens.includes("cotizacion") || tokens.includes("cotizaciones");
    const tablaDeseada = quiereCotizacion ? "cotizaciones" : "ordenes";

    if (idCli && (kw === "orden" || kw === "ordenes" || kw === "cotizacion" || kw === "cotizaciones")) {
      const ok = await flujoOrdenesPorIdCliente({ idCliente: idCli, tabla: tablaDeseada, original, tokens });
      saveSearch(original);
      renderRecents();
      return ok;
    }

    // Handlers dinámicos (capacitaciones/cursos)
    if (API_HANDLERS[kw]) {
      const result = await API_HANDLERS[kw]({ original, qNorm, tokens, kw });
      if (result?.redirectUrl) window.open(result.redirectUrl, "_blank");
      saveSearch(original);
      renderRecents();
      return;
    }

    // Flujo genérico: resolver cliente (RFC/nombre) → traer órdenes/cotizaciones
    const ok = await flujoClienteYOrdenes({ original, qNorm, tokens, tabla: tablaDeseada });
    saveSearch(original);
    renderRecents();
    return ok;
  }

  // ================== HANDLER CAPACITACIONES ==================
  async function handlerCapacitaciones(ctx) {
    const { original, qNorm, tokens } = ctx;
    const idCli = extractIdCliente(qNorm);
    const { fechaini, fechafin } = extractRangoFechas(original);

    if (idCli) {
      await flujoCapacitacionesPorIdCliente({ idCliente: idCli, original, fechaini, fechafin });
      return { handled: true };
    }

    const clienteHint = extractClienteNombre(original);
    const rfc = extractRFC(original);

    let nombre = clienteHint;
    if (!nombre && (tokens.includes("cliente") || tokens.includes("clientes"))) {
      nombre = extractTextAfter(original, /(cliente|clientes)\s+/i, ["orden","ordenes","cotizacion","cotizaciones","fecha","rfc","id","id cliente","curso","cursos","capacitacion","capacitaciones","inicial","final","inicio","fin"]);
    }

    const res = await postForm(ENDPOINTS.buscarCliente, {
      ...GLOBALS,
      ...(rfc ? { rfc } : {}),
      ...(nombre ? { nombre } : {}),
    });

    const data = res?.data;
    const clientes = Array.isArray(data) ? data : (data ? [data] : []);

    if (clientes.length === 0) {
      showModal("Sin resultados", `<p class="mb-0">No se encontraron clientes con ${rfc ? "el RFC" : "el nombre"} proporcionado.</p>`);
      return { handled: true };
    }

    if (clientes.length === 1) {
      const idCliente = clientes[0]?.IdCliente;
      if (!idCliente) {
        showModal("Error", `<p class="mb-0">Respuesta de cliente inválida (sin IdCliente).</p>`);
        return { handled: true };
      }
      await flujoCapacitacionesPorIdCliente({ idCliente, original, fechaini, fechafin });
      return { handled: true };
    }

    const filas = clientes.map((c, idx) => {
      const searchText = normalizeSearch([c.IdCliente, c.Nombre, c.RFC].join(" "));
      return `
        <tr data-search="${escapeAttr(searchText)}">
          <td class="xtf-hcell">#${idx + 1}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.IdCliente ?? ""))}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.Nombre ?? ""))}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.RFC ?? ""))}</td>
          <td class="text-end">
            <button class="btn btn-sm btn-primary" data-choose-id="${escapeAttr(String(c.IdCliente))}">Ver capacitaciones</button>
          </td>
        </tr>
      `;
    }).join("");

    const cuerpo = `
      <div class="order-search position-relative my-3">
        <input class="form-control rounded-5 px-5 xtf-search-input" type="text" placeholder="Buscar clientes (multi-palabra e IDs)">
        <span class="material-icons-outlined position-absolute ms-3 translate-middle-y start-0 top-50">search</span>
      </div>
      <div class="table-responsive">
        <table class="table align-middle">
          <thead>
            <tr>
              <th>#</th>
              <th>IdCliente</th>
              <th>Nombre</th>
              <th>RFC</th>
              <th class="text-end">Acción</th>
            </tr>
          </thead>
          <tbody class="xtf-tbody">${filas}</tbody>
        </table>
      </div>
      <div class="xtf-no-results text-center text-muted py-3 d-none">Sin resultados</div>
    `;
    const { modalEl } = showModal("Selecciona un cliente", plantillaCard("Coincidencias de clientes", cuerpo));

    modalEl.addEventListener("click", async (e) => {
      const btn = e.target.closest("button[data-choose-id]");
      if (!btn) return;
      const idCliente = btn.getAttribute("data-choose-id");
      await flujoCapacitacionesPorIdCliente({ idCliente, original, fechaini, fechafin });
    });

    attachTableFilter(modalEl);
    return { handled: true };
  }

  // ================== FLUJO CAPACITACIONES ==================
  async function flujoCapacitacionesPorIdCliente({ idCliente, original, fechaini, fechafin }) {
    const rango = extractRangoFechas(original);
    const Fechaini = fechaini || rango.fechaini || undefined;
    const Fechafin = fechafin || rango.fechafin || undefined;

    const res = await postForm(ENDPOINTS.buscarEventos, {
      ...GLOBALS,
      IdCliente: idCliente,
      ...(Fechaini ? { Fechaini } : {}),
      ...(Fechafin ? { Fechafin } : {}),
    });

    const lista = Array.isArray(res?.data) ? res.data : [];
    if (lista.length === 0) {
      showModal("Sin resultados", `<p class=\"mb-0\">No se encontraron <b>capacitaciones</b> para el cliente ${escapeHTML(String(idCliente))}${Fechaini || Fechafin ? ` en el rango ${escapeHTML(Fechaini || "…")} a ${escapeHTML(Fechafin || "…")}` : ""}.</p>`);
      return false;
    }

    if (lista.length === 1 && Number(lista[0]?.IdOrden) > 0) {
      const id = Number(lista[0].IdOrden);
      const url = `${BASE_URL}/${ROUTES.orden}/${encodeURIComponent(id)}`;
      window.open(url, "_blank");
      return true;
    }

    const filas = lista.map((e) => {
      const idEvt  = e?.id ?? e?.evntId ?? "";
      const ini    = formatEventDate(e?.evntFechaInicio, e?.evntHoraInicio, e?.start);
      const fin    = formatEventDate(e?.evntFechaFin, e?.evntHoraFin, e?.end);
      const serv   = e?.evntTipoServicio || e?.title || "";
      const idOrd  = e?.IdOrden ?? "";
      const folio  = e?.FolioOrden ?? "";
      const nombre = e?.Nombre ?? "";
      const obs    = e?.evtObservaciones ?? e?.description ?? "";
      const searchText = normalizeSearch([idEvt, ini, fin, serv, idOrd, folio, nombre, obs].join(" "));
      const btn = (Number(idOrd) > 0)
        ? `<button class="btn btn-sm btn-danger" data-open-orden="${escapeAttr(String(idOrd))}">Abrir</button>`
        : `<button class="btn btn-sm btn-secondary" disabled title="Sin orden asociada">Sin orden</button>`;
      return `
        <tr data-search="${escapeAttr(searchText)}">
          <td class="xtf-hcell">${escapeHTML(String(idEvt))}</td>
          <td class="xtf-hcell">${escapeHTML(ini)}</td>
          <td class="xtf-hcell">${escapeHTML(fin)}</td>
          <td class="xtf-hcell">${escapeHTML(serv)}</td>
          <td class="xtf-hcell">${escapeHTML(String(folio || idOrd || "—"))}</td>
          <td class="xtf-hcell">${escapeHTML(nombre)}</td>
          <td class="xtf-hcell">${escapeHTML(obs)}</td>
          <td class="text-end">${btn}</td>
        </tr>
      `;
    }).join("");

    const cuerpo = `
      <div class="order-search position-relative my-3">
        <input class="form-control rounded-5 px-5 xtf-search-input" type="text" placeholder="Buscar capacitaciones (multi-palabra e IDs)">
        <span class="material-icons-outlined position-absolute ms-3 translate-middle-y start-0 top-50">search</span>
      </div>
      <div class="table-responsive">
        <table class="table align-middle">
          <thead>
            <tr>
              <th>ID Evento</th>
              <th>Inicio</th>
              <th>Fin</th>
              <th>Servicio</th>
              <th>Orden</th>
              <th>Cliente</th>
              <th>Observaciones</th>
              <th class="text-end">Acción</th>
            </tr>
          </thead>
          <tbody class="xtf-tbody">${filas}</tbody>
        </table>
      </div>
      <div class="xtf-no-results text-center text-muted py-3 d-none">Sin resultados</div>
    `;

    const { modalEl } = showModal(
      "Capacitaciones",
      plantillaCard("Capacitaciones del cliente", cuerpo)
    );

    modalEl.addEventListener("click", (e) => {
      const btn = e.target.closest("button[data-open-orden]");
      if (!btn) return;
      const id = btn.getAttribute("data-open-orden");
      if (id) {
        const url = `${BASE_URL}/${ROUTES.orden}/${encodeURIComponent(id)}`;
        window.open(url, "_blank");
      }
    });

    attachTableFilter(modalEl);
    return true;
  }

  // ================== FLUJOS ORDENES/COTIZACIONES ==================
  async function flujoOrdenesPorIdCliente({ idCliente, tabla, original }) {
    const fecha = extractFecha(original);
    const res = await postForm(ENDPOINTS.buscarOrdenes, {
      ...GLOBALS,
      IdCliente: idCliente,
      tabla,
      valtabla: tabla,
      ...(fecha ? { fecha } : {})
    });

    const lista = Array.isArray(res?.data) ? res.data : [];
    if (lista.length === 0) {
      showModal("Sin resultados", `<p class="mb-0">No se encontraron ${tabla} para el cliente ${escapeHTML(String(idCliente))}${fecha ? " en la fecha " + escapeHTML(fecha) : ""}.</p>`);
      return false;
    }

    if (lista.length === 1) {
      const unico = lista[0];
      const id = getItemId(unico, tabla);
      if (id) {
        const route = (tabla === "cotizaciones") ? ROUTES.cotizacion : ROUTES.orden;
        const url = `${BASE_URL}/${route}/${encodeURIComponent(id)}`;
        window.open(url, "_blank");
        return true;
      }
    }

    const filas = lista.map((o) => {
      const id    = getItemId(o, tabla);
      const fecha = safe(o, "Fecha", "");
      const estatus = safe(o, "NombreEstatus", "");
      const tipo    = safe(o, "NombreTipoOrden", "");
      const trans   = safe(o, "DeptoDescripcion", "");
      const searchText = normalizeSearch([id, fecha, estatus, tipo, trans].join(" "));
      return `
        <tr data-search="${escapeAttr(searchText)}">
          <td class="xtf-hcell">${escapeHTML(String(id ?? ""))}</td>
          <td class="xtf-hcell">${escapeHTML(String(formateaFecha(fecha)))}</td>
          <td class="xtf-hcell">${escapeHTML(String(estatus))}</td>
          <td class="xtf-hcell">${escapeHTML(String(tipo))}</td>
          <td class="xtf-hcell">${escapeHTML(String(trans))}</td>
          <td class="text-end">
            <button class="btn btn-sm btn-danger" data-open-id="${escapeAttr(String(id ?? ""))}" data-tabla="${escapeAttr(tabla)}">Abrir</button>
          </td>
        </tr>
      `;
    }).join("");

    const idHeader = (tabla === "cotizaciones") ? "ID Cotización" : "ID Orden";

    const cuerpo = `
      <div class="order-search position-relative my-3">
        <input class="form-control rounded-5 px-5 xtf-search-input" type="text" placeholder="Buscar en resultados (multi-palabra e IDs)">
        <span class="material-icons-outlined position-absolute ms-3 translate-middle-y start-0 top-50">search</span>
      </div>
      <div class="table-responsive">
        <table class="table align-middle">
          <thead>
            <tr>
              <th>${idHeader}</th>
              <th>Fecha</th>
              <th>Estatus</th>
              <th>Tipo de orden</th>
              <th>Transito</th>
              <th class="text-end">Acción</th>
            </tr>
          </thead>
          <tbody class="xtf-tbody">${filas}</tbody>
        </table>
      </div>
      <div class="xtf-no-results text-center text-muted py-3 d-none">Sin resultados</div>
    `;

    const { modalEl } = showModal(
      "Resultados",
      plantillaCard(
        (tabla === "cotizaciones" ? "Cotizaciones" : "Órdenes") + " del cliente",
        cuerpo
      )
    );

    modalEl.addEventListener("click", (e) => {
      const btn = e.target.closest("button[data-open-id]");
      if (!btn) return;
      const id = btn.getAttribute("data-open-id");
      const t  = btn.getAttribute("data-tabla");
      const route = (t === "cotizaciones") ? ROUTES.cotizacion : ROUTES.orden;
      if (id && route) {
        const url = `${BASE_URL}/${route}/${encodeURIComponent(id)}`;
        window.open(url, "_blank");
      }
    });

    attachTableFilter(modalEl);
    return true;
  }

  async function flujoClienteYOrdenes({ original, qNorm, tokens, tabla }) {
    const clienteHint = extractClienteNombre(original);
    const rfc = extractRFC(original);
    const fecha = extractFecha(original);

    let nombre = clienteHint;
    if (!nombre && (tokens.includes("cliente") || tokens.includes("clientes"))) {
      nombre = extractTextAfter(original, /(cliente|clientes)\s+/i, ["orden","ordenes","cotizacion","cotizaciones","fecha","rfc","id","id cliente","curso","cursos","capacitacion","capacitaciones"]);
    }

    const res = await postForm(ENDPOINTS.buscarCliente, {
      ...GLOBALS,
      ...(rfc ? { rfc } : {}),
      ...(nombre ? { nombre } : {}),
      ...(fecha ? { fecha } : {})
    });

    const data = res?.data;
    const clientes = Array.isArray(data) ? data : (data ? [data] : []);

    if (clientes.length === 0) {
      showModal("Sin resultados", `<p class="mb-0">No se encontraron clientes con ${rfc ? "el RFC" : "el nombre"} proporcionado.</p>`);
      return false;
    }

    if (clientes.length === 1) {
      const idCliente = clientes[0]?.IdCliente;
      if (!idCliente) {
        showModal("Error", `<p class="mb-0">Respuesta de cliente inválida (sin IdCliente).</p>`);
        return false;
      }
      return await flujoOrdenesPorIdCliente({ idCliente, tabla, original, tokens });
    }

    const filas = clientes.map((c, idx) => {
      const searchText = normalizeSearch([c.IdCliente, c.Nombre, c.RFC].join(" "));
      return `
        <tr data-search="${escapeAttr(searchText)}">
          <td class="xtf-hcell">#${idx + 1}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.IdCliente ?? ""))}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.Nombre ?? ""))}</td>
          <td class="xtf-hcell">${escapeHTML(String(c.RFC ?? ""))}</td>
          <td class="text-end">
            <button class="btn btn-sm btn-primary" data-choose-id="${escapeAttr(String(c.IdCliente))}">Ver ${tabla}</button>
          </td>
        </tr>
      `;
    }).join("");

    const cuerpo = `
      <div class="order-search position-relative my-3">
        <input class="form-control rounded-5 px-5 xtf-search-input" type="text" placeholder="Buscar clientes (multi-palabra e IDs)">
        <span class="material-icons-outlined position-absolute ms-3 translate-middle-y start-0 top-50">search</span>
      </div>
      <div class="table-responsive">
        <table class="table align-middle">
          <thead>
            <tr>
              <th>#</th>
              <th>IdCliente</th>
              <th>Nombre</th>
              <th>RFC</th>
              <th class="text-end">Acción</th>
            </tr>
          </thead>
          <tbody class="xtf-tbody">${filas}</tbody>
        </table>
      </div>
      <div class="xtf-no-results text-center text-muted py-3 d-none">Sin resultados</div>
    `;

    const { modalEl } = showModal("Selecciona un cliente", plantillaCard("Coincidencias de clientes", cuerpo));
    modalEl.addEventListener("click", async (e) => {
      const btn = e.target.closest("button[data-choose-id]");
      if (!btn) return;
      const idCliente = btn.getAttribute("data-choose-id");
      await flujoOrdenesPorIdCliente({ idCliente, tabla, original, tokens });
    });

    attachTableFilter(modalEl);
    return true;
  }

  // ================== FILTRO + HIGHLIGHT ==================
  function attachTableFilter(modalEl) {
    const input = modalEl.querySelector(".xtf-search-input");
    const tbody = modalEl.querySelector(".xtf-tbody");
    const noRes = modalEl.querySelector(".xtf-no-results");
    if (!input || !tbody || !noRes) return;

    const rows = Array.from(tbody.querySelectorAll("tr"));

    // Guardar texto crudo y data-search
    rows.forEach(r => {
      const tds = Array.from(r.querySelectorAll("td"));
      tds.forEach((td, idx) => {
        if (idx === tds.length - 1) return; // omite acción
        if (!td.dataset.raw) td.dataset.raw = td.textContent || "";
      });
      if (!r.hasAttribute("data-search")) r.setAttribute("data-search", normalizeSearch(r.textContent || ""));
    });

    const doFilter = () => {
      const tokens = normalizeSearch(input.value).split(" ").filter(Boolean);

      if (tokens.length === 0) {
        rows.forEach(r => r.style.display = "");
        noRes.classList.add("d-none");
      } else {
        let visibleCount = 0;
        rows.forEach(r => {
          const h = r.getAttribute("data-search") || "";
          const match = tokens.every(t => h.includes(t));
          r.style.display = match ? "" : "none";
          if (match) visibleCount++;
        });
        (visibleCount === 0) ? noRes.classList.remove("d-none") : noRes.classList.add("d-none");
      }

      // Highlight
      rows.forEach(r => {
        const display = r.style.display !== "none";
        const cells = Array.from(r.querySelectorAll("td"));
        cells.forEach((td, idx) => {
          if (idx === cells.length - 1) return; // omite acción
          const raw = td.dataset.raw ?? td.textContent ?? "";
          if (!display || tokens.length === 0) {
            td.innerHTML = escapeHTML(raw);
          } else {
            td.innerHTML = highlightWithTokens(raw, tokens);
          }
        });
      });
    };

    input.addEventListener("input", doFilter);
  }

  // ---- Highlight helpers ----
  function highlightWithTokens(text, tokensNorm) {
    if (!text) return "";
    const { norm, map } = buildNormMap(text);
    let ranges = [];
    tokensNorm.forEach(t => {
      if (!t) return;
      let from = 0;
      while (true) {
        const pos = norm.indexOf(t, from);
        if (pos === -1) break;
        const start = map[pos];
        const end   = map[pos + t.length - 1] + 1; // exclusivo
        if (start != null && end != null) ranges.push([start, end]);
        from = pos + t.length;
      }
    });
    if (ranges.length === 0) return escapeHTML(text);
    ranges.sort((a,b) => a[0]-b[0] || a[1]-b[1]);
    const merged = [];
    let cur = ranges[0];
    for (let i=1;i<ranges.length;i++){
      const r = ranges[i];
      if (r[0] <= cur[1]) cur[1] = Math.max(cur[1], r[1]);
      else { merged.push(cur); cur = r; }
    }
    merged.push(cur);
    let out = "", idx = 0;
    merged.forEach(([s,e]) => {
      out += escapeHTML(text.slice(idx, s));
      out += "<mark>" + escapeHTML(text.slice(s, e)) + "</mark>";
      idx = e;
    });
    out += escapeHTML(text.slice(idx));
    return out;
  }

  function buildNormMap(str) {
    let norm = "";
    const map = [];
    for (let i = 0; i < str.length; i++) {
      let out = str[i].toLowerCase();
      out = out.normalize("NFD").replace(/[\u0300-\u036f]/g, "");
      if (/^[a-z0-9]$/.test(out)) { norm += out; map.push(i); }
      else if (/\s/.test(out))    { norm += " "; map.push(i); }
      else                        { norm += " "; map.push(i); }
    }
    return { norm, map };
  }

  // ================== HELPERS QUERY ==================
  function detectarKeyword(tokens) {
    for (let i = 0; i < tokens.length; i++) {
      const cur = tokens[i];
      const next = tokens[i + 1] || "";
      if ((cur === "id" && next === "cliente") || cur === "id cliente") return "id cliente";
      if (KEYWORDS.includes(cur)) return cur;
    }
    return null;
  }

  function extractId(text) {
    const m = text.match(/(^|\s)(\d{2,})(\s|$)/);
    return m ? m[2] : null;
  }

  function extractIdCliente(text) {
    const m = text.match(/id\s+cliente\s+(\d{1,})/i);
    return m ? m[1] : null;
  }

  function extractRFC(original) {
    const m = original.match(/\brfc\s+([a-z0-9\-]+)/i);
    return m ? m[1] : null;
  }

  function extractFecha(original) {
    const m1 = original.match(/\b(\d{4})-(\d{2})-(\d{2})\b/);
    if (m1) return m1[0];
    const m2 = original.match(/\b(\d{2})\/(\d{2})\/(\d{4})\b/);
    if (m2) return `${m2[3]}-${m2[2]}-${m2[1]}`;
    return null;
  }

  function extractClienteNombre(original) {
    return extractTextAfter(original, /\b(cliente|clientes)\s+/i, ["orden","ordenes","cotizacion","cotizaciones","fecha","rfc","id","id cliente","curso","cursos","capacitacion","capacitaciones","inicial","final","inicio","fin"]);
  }

  function extractTextAfter(text, regexStart, stopKeywords = []) {
    const idx = text.search(regexStart);
    if (idx === -1) return null;
    const start = text.slice(idx).replace(regexStart, "");
    if (!stopKeywords || stopKeywords.length === 0) return start.trim();
    const stopRegex = new RegExp(`\\b(${stopKeywords.map(escapeReg).join("|")})\\b`, "i");
    const stop = start.search(stopRegex);
    return (stop === -1 ? start : start.slice(0, stop)).trim();
  }

  function extractRangoFechas(text) {
    const dx = parseDate;
    const ini = (
      text.match(/\b(fecha\s*inicial|fechaini|inicio)\s+([0-9]{4}-[0-9]{2}-[0-9]{2}|[0-9]{2}\/[0-9]{2}\/[0-9]{4})/i)?.[2]
    );
    const fin = (
      text.match(/\b(fecha\s*final|fechafin|fin)\s+([0-9]{4}-[0-9]{2}-[0-9]{2}|[0-9]{2}\/[0-9]{2}\/[0-9]{4})/i)?.[2]
    );
    return {
      fechaini: ini ? dx(ini) : null,
      fechafin: fin ? dx(fin) : null
    };
  }

  function parseDate(s) {
    if (!s) return null;
    const iso = s.match(/^(\d{4})-(\d{2})-(\d{2})$/);
    if (iso) return iso[0];
    const lat = s.match(/^(\d{2})\/(\d{2})\/(\d{4})$/);
    if (lat) return `${lat[3]}-${lat[2]}-${lat[1]}`;
    return null;
  }

  function escapeReg(s) { return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); }

  function normalize(str) {
    return (str || "")
      .toLowerCase()
      .normalize("NFD").replace(/[\u0300-\u036f]/g, "")
      .replace(/\s+/g, " ")
      .trim();
  }

  // === Normalización para BUSCADOR (mantiene dígitos, ignora símbolos)
  function normalizeSearch(str) {
    return String(str || "")
      .toLowerCase()
      .normalize("NFD")
      .replace(/[\u0300-\u036f]/g, "")
      .replace(/[^a-z0-9\s]/g, " ")
      .replace(/\s+/g, " ")
      .trim();
  }

  // ================== HELPERS FETCH ==================
  async function postForm(url, params) {
    const body = new URLSearchParams();
    Object.entries(params || {}).forEach(([k, v]) => {
      if (v !== undefined && v !== null) body.append(k, String(v));
    });

    const res = await fetch(url, {
      method: "POST",
      headers: { "Content-Type": "application/x-www-form-urlencoded; charset=UTF-8" },
      body
    });

    let json = {};
    try { json = await res.json(); } catch (_e) {}
    if (!res.ok) {
      const msg = json?.msg || json?.message || res.statusText;
      throw new Error(`POST ${url} → ${res.status} ${msg}`);
    }
    return json;
  }

  // ================== MODAL ==================
  function plantillaCard(titulo, innerHTML) {
    return `
      <div class="col-lg-12 col-xxl-12 d-flex align-items-stretch">
        <div class="card w-100 rounded-4">
          <div class="card-body">
            <div class="d-flex align-items-start justify-content-between mb-3">
              <div><h5 class="mb-0">${escapeHTML(titulo)}</h5></div>
              <div>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
              </div>
            </div>
            ${innerHTML}
          </div>
        </div>
      </div>`;
  }

  function ensureModal() {
    let modalEl = document.getElementById("xtf-busca-modal");
    if (!modalEl) {
      modalEl = document.createElement("div");
      modalEl.id = "xtf-busca-modal";
      modalEl.className = "modal fade";
      modalEl.innerHTML = `
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content rounded-4">
            <div class="modal-header">
              <h5 class="modal-title">Resultados</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body"><p class="mb-0">Procesando…</p></div>
          </div>
        </div>`;
      document.body.appendChild(modalEl);
    }
    return modalEl;
  }

  function showModal(titulo, htmlCuerpo) {
    const modalEl = ensureModal();
    modalEl.querySelector(".modal-title").textContent = titulo || "Resultados";
    modalEl.querySelector(".modal-body").innerHTML = htmlCuerpo || "";

    let inst = null;
    if (window.bootstrap && bootstrap.Modal) {
      inst = bootstrap.Modal.getOrCreateInstance(modalEl, { backdrop: true });
      inst.show();
    } else {
      modalEl.style.display = "block";
      modalEl.classList.add("show");
      modalEl.addEventListener("click", (e) => {
        if (e.target === modalEl) { modalEl.style.display = "none"; modalEl.classList.remove("show"); }
      }, { once: true });
    }
    return { modalEl, instance: inst };
  }

  // ================== BÚSQUEDAS RECIENTES ==================
  function getRecents() {
    try {
      const raw = localStorage.getItem(STORAGE_KEY);
      const arr = raw ? JSON.parse(raw) : [];
      return Array.isArray(arr) ? arr : [];
    } catch { return []; }
  }

  function saveSearch(text) {
    const rec = getRecents();
    const cleaned = rec.filter(v => v.trim().toLowerCase() !== text.trim().toLowerCase());
    cleaned.unshift(text);
    if (cleaned.length > MAX_RECENTS) cleaned.length = MAX_RECENTS;
    localStorage.setItem(STORAGE_KEY, JSON.stringify(cleaned));
  }

  function renderRecents() {
    const cont = document.getElementById(RECENTS_ID);
    if (!cont) return;
    const recents = getRecents();
    cont.innerHTML = recents.map(txt => (
      `<a href="javascript:;" class="kewords"><span>${escapeHTML(txt)}</span><i class="material-icons-outlined fs-6">search</i></a>`
    )).join("");
  }

  // === NUEVO: Toast helper (usa tu configuración) ===
  function toastFire(opts) {
    try {
      const T = window.Toast || (window.Swal && Swal.mixin ? Swal.mixin({
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true,
        didOpen: (toast) => {
          toast.onmouseenter = Swal.stopTimer;
          toast.onmouseleave = Swal.resumeTimer;
        }
      }) : null);
      if (T && T.fire) {
        T.fire({ icon: opts.icon || "info", title: opts.title || "", theme: "dark" });
        // guardar la instancia para siguientes usos
        if (!window.Toast) window.Toast = T;
      }
    } catch (_) {}
  }

  // === NUEVO: Confirmación para borrar búsquedas (SweetAlert2) ===
  async function confirmClearSearches() {
    if (!window.Swal || !Swal.fire) {
      // Si no existe SweetAlert2, borra directo y muestra toast simple (si hay)
      try { clearSearches(); } catch (e) {}
      toastFire({ icon: "success", title: "Búsquedas borradas" });
      return;
    }
    try {
      const result = await Swal.fire({
        title: "¿Borrar búsquedas?",
        text: "Se eliminarán todas las búsquedas recientes.",
        icon: "warning",
        showCancelButton: true,
        confirmButtonColor: '#DC3545',
        confirmButtonText: "Sí, borrar",
        cancelButtonText: "Cancelar",
        theme: 'dark',
        reverseButtons: true
      });
      if (result.isConfirmed) {
        clearSearches();
        toastFire({ icon: "success", title: "Búsquedas borradas" });
      } else if (result.dismiss === Swal.DismissReason.cancel) {
        toastFire({ icon: "info", title: "Operación cancelada" });
      }
    } catch (err) {
      toastFire({ icon: "error", title: "Hubo un error en la aplicación" });
    }
  }

  // === Limpiar todas las búsquedas recientes ===
  function clearSearches() {
    try {
      localStorage.removeItem(STORAGE_KEY);
    } catch {}
    renderRecents();
  }
  // Exponer utilidades globales
  window.limpiarBusquedasXT = clearSearches;
  window.confirmarBorradoBusquedasXT = confirmClearSearches;

  // ================== UTILS ==================
  function getItemId(item, tabla) {
    if (tabla === "cotizaciones") {
      return item?.IdCotizacion ?? item?.IdOrden ?? null;
    }
    return item?.IdOrden ?? item?.IdCotizacion ?? null;
  }

  function formatEventDate(fecha, hora, iso) {
    if (fecha) {
      const hhmm = (hora || "").slice(0,5);
      return `${fecha}${hhmm ? " " + hhmm : ""}`;
    }
    if (iso) {
      const m = String(iso).match(/^(\d{4}-\d{2}-\d{2})[T ](\d{2}:\d{2})/);
      return m ? `${m[1]} ${m[2]}` : iso;
    }
    return "";
  }

  function formateaFecha(s) {
    if (!s) return "";
    const m = String(s).match(/^(\d{4}-\d{2}-\d{2})\s+(\d{2}:\d{2})/);
    return m ? `${m[1]} ${m[2]}` : s;
  }

  function safe(obj, key, def) {
    return (obj && key in obj && obj[key] !== null && obj[key] !== undefined) ? obj[key] : def;
  }

  function escapeHTML(s) {
    return String(s)
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#039;");
  }
  function escapeAttr(s) { return escapeHTML(s); }

})();