/* reimpresion-modal.js (v2)
   Flujo:
   - IdOrden (numérico): POST codi_get_one_orden { IdOrden } → muestra acciones.
   - RFC (alfa-numérico con dígitos): POST codi_get_one_cliente { rfc } → lista clientes → elegir → POST codi_get_one_orden { IdCliente, tabla:'ordenes' } → elegir orden → acciones.
   - Nombre (solo letras/espacios): POST codi_get_one_cliente { nombre } → (mismo flujo que RFC).
*/

(() => {
  const API_BASE = 'https://grupoxtinfire.com/admin/ajax/';
  const HOOK = 'codi_hook';
  const ACTION = 'load';
  const ENDPOINTS = {
    cliente: 'codi_get_one_cliente', // { rfc?, nombre? }
    orden:   'codi_get_one_orden'    // { IdOrden? } ó { IdCliente, tabla:'ordenes' }
  };

  // ---- Helpers DOM / HTTP ----
  const $ = (s, c=document) => c.querySelector(s);
  const el = (tag, attrs, ...kids) => {
    const n = document.createElement(tag);
    if (attrs) for (const [k,v] of Object.entries(attrs)) {
      if (v==null || v===false) continue;
      if (k==='class') n.className = v;
      else if (k.startsWith('on') && typeof v==='function') n.addEventListener(k.slice(2), v);
      else n.setAttribute(k, v);
    }
    kids.flat().forEach(k => {
      if (k==null || k===false) return;
      n.appendChild(typeof k==='string' || typeof k==='number' ? document.createTextNode(String(k)) : k);
    });
    return n;
  };
  const spinner = () => el('div',{class:'d-flex justify-content-center my-3'},
    el('div',{class:'spinner-border text-danger me-2',role:'status'}),'Cargando...'
  );
  const apiPost = async (fn, params={}) => {
    const body = new URLSearchParams({ hook: HOOK, action: ACTION, ...params });
    const r = await fetch(API_BASE+fn, { method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8'}, body });
    if (!r.ok) throw new Error(`HTTP ${r.status}`);
    const j = await r.json();
    if (!j || j.status!==200 || j.error) throw new Error(j?.msg || 'Error API');
    return j.data || [];
  };

  // ---- Detección de tipo de entrada ----
  const isOnlyDigits = v => /^\d+$/.test(v.trim());
  const hasAnyDigit  = v => /\d/.test(v);               // RFC contiene dígitos
  const isLettersAndSpaces = v => /^[A-Za-zÁÉÍÓÚÑáéíóúñ\s]+$/.test(v.trim());

  // ---- Render acciones finales (ticket/factura/xml) ----
  const renderAccionesDeOrden = (container, orden, clienteCtx=null) => {
    container.innerHTML = '';

    // intentar mostrar nombre de cliente si viene en la orden (por si el back lo entrega)
    const nombreCliente = clienteCtx?.Nombre || orden?.NombreCliente || orden?.Cliente || '(No disponible)';

    const header = el('div',{class:'mb-3'},
      el('div',{},
        el('span',{class:'badge text-bg-danger me-2'}, `Orden #${orden.IdOrden}`),
        el('span',{class:'badge text-bg-secondary'}, orden.NombreTipoOrden || '')
      ),
      el('div',{class:'mt-2'},
        el('div',{}, el('strong',{},'Cliente: '), nombreCliente),
        el('div',{}, el('strong',{},'Fecha: '), orden.Fecha || '')
      )
    );

    const actions = el('div',{class:'d-flex flex-wrap gap-2'},
      el('a',{
        class:'btn btn-danger',
        href:`https://grupoxtinfire.com/admin/pdf/pdfticket/${orden.IdOrden}`,
        target:'_blank', rel:'noopener'
      },'Reimprimir ticket'),
      (() => {
        const b = el('button',{class:'btn btn-outline-danger',type:'button'},'Reimprimir factura');
        if (orden.IdFactura) {
          b.addEventListener('click',()=> {
            if (typeof buscarAcusePdf==='function') buscarAcusePdf(orden.IdFactura);
            else alert('Función buscarAcusePdf no disponible.');
          });
        } else { b.disabled = true; b.title = 'Orden sin factura'; }
        return b;
      })(),
      (() => {
        const b = el('button',{class:'btn btn-outline-danger',type:'button'},'XML de factura');
        if (orden.IdFactura) {
          b.addEventListener('click',()=> {
            if (typeof buscarAcuseXml==='function') buscarAcuseXml(orden.IdFactura);
            else alert('Función buscarAcuseXml no disponible.');
          });
        } else { b.disabled = true; b.title = 'Orden sin factura'; }
        return b;
      })()
    );

    container.appendChild(header);
    container.appendChild(actions);
  };

  // ---- Búsqueda directa por IdOrden ----
  const buscarPorIdOrden = async (container, idOrden) => {
    container.innerHTML = ''; const sp = spinner(); container.appendChild(sp);
    let data = [];
    try {
      data = await apiPost(ENDPOINTS.orden, { IdOrden: Number(idOrden) });
    } catch (e) {
      console.error('Error IdOrden:', e);
    }
    sp.remove();

    if (!data.length) {
      container.appendChild(el('div',{class:'alert alert-warning'}, `No existe la orden #${idOrden}.`));
      return;
    }
    // Si el back devuelve una lista, tomamos la que coincida exactamente:
    const o = data.find(x=> Number(x.IdOrden)===Number(idOrden)) || data[0];
    renderAccionesDeOrden(container, o, null);
  };

  // ---- Lista de clientes (RFC o Nombre) ----
  const renderClientes = (container, clientes=[]) => {
    container.innerHTML = '';
    if (!clientes.length) {
      container.appendChild(el('div',{class:'alert alert-warning'},'No se encontraron clientes con ese criterio.'));
      return;
    }
    container.appendChild(el('h6',{},'Selecciona un cliente'));
    const list = el('div',{class:'list-group'});
    clientes.forEach(cli=>{
      const row = el('div',{class:'list-group-item d-flex justify-content-between align-items-center'},
        el('div',{},
          el('div',{class:'fw-semibold'}, cli.Nombre || '(Sin nombre)'),
          el('div',{class:'small text-muted'},
            (cli.RFC||'') + (cli.FisEmail? ' · '+cli.FisEmail:'') + (cli.FisTelefono? ' · '+cli.FisTelefono:'')
          )
        ),
        (()=> {
          const b = el('button',{class:'btn btn-sm btn-danger'},'Elegir');
          b.addEventListener('click',(e)=>{ e.preventDefault(); e.stopPropagation(); cargarOrdenesDeCliente(container, cli); });
          return b;
        })()
      );
      list.appendChild(row);
    });
    container.appendChild(list);
  };

  // ---- Órdenes de un cliente seleccionado ----
  const cargarOrdenesDeCliente = async (container, cliente) => {
    container.innerHTML = '';
    container.appendChild(
      el('div',{class:'mb-2'},
        el('span',{class:'badge text-bg-danger'},'Cliente seleccionado: '),' ',
        cliente.Nombre || ''
      )
    );

    const sp = spinner(); container.appendChild(sp);
    let ordenes = [];
    try {
      // EXACTO: IdCliente + tabla='ordenes'
      ordenes = await apiPost(ENDPOINTS.orden, { IdCliente: cliente.IdCliente, tabla: 'ordenes' });
    } catch (e) {
      console.error('Error órdenes por IdCliente:', e);
    }
    sp.remove();

    if (!ordenes.length) {
      container.appendChild(el('div',{class:'alert alert-warning'},'Este cliente no tiene órdenes registradas.'));
      return;
    }

    container.appendChild(el('h6',{},'Órdenes de este cliente'));
    const tbl = el('table',{class:'table table-sm align-middle'},
      el('thead',{}, el('tr',{},
        el('th',{},'Id'),
        el('th',{},'Fecha'),
        el('th',{},'Tipo'),
        el('th',{},'Estatus'),
        el('th',{},'Depto'),
        el('th',{},'Factura'),
        el('th',{},'Acciones')
      )),
      el('tbody',{})
    );

    ordenes.forEach(o=>{
      const tr = el('tr',{},
        el('td',{}, o.IdOrden),
        el('td',{}, o.Fecha || ''),
        el('td',{}, o.NombreTipoOrden || ''),
        el('td',{}, o.NombreEstatus || ''),
        el('td',{}, o.DeptoDescripcion || ''),
        el('td',{}, o.IdFactura ? el('span',{class:'badge text-bg-danger'},'Sí') : el('span',{class:'badge text-bg-secondary'},'No')),
        el('td',{}, el('button',{
          class:'btn btn-sm btn-danger',
          onclick:()=> renderAccionesDeOrden(container, o, cliente)
        },'Seleccionar'))
      );
      tbl.tBodies[0].appendChild(tr);
    });

    container.appendChild(tbl);
  };

  // ---- Búsqueda principal ----
  const doSearch = async () => {
    const q = $('#rei-input').value.trim();
    const results = $('#rei-results');
    results.innerHTML = '';
    if (!q) { results.appendChild(el('div',{class:'alert alert-warning'},'Escribe RFC, Nombre o IdOrden.')); return; }

    // 1) IdOrden directo (solo dígitos)
    if (isOnlyDigits(q)) { await buscarPorIdOrden(results, q); return; }

    // 2) RFC (contiene dígitos)
    if (hasAnyDigit(q)) {
      const sp = spinner(); results.appendChild(sp);
      let clientes = [];
      try { clientes = await apiPost(ENDPOINTS.cliente, { rfc: q }); } catch(e){ console.error(e); }
      results.innerHTML = '';
      renderClientes(results, clientes);
      return;
    }

    // 3) Nombre (solo letras y espacios)
    if (isLettersAndSpaces(q)) {
      const sp = spinner(); results.appendChild(sp);
      let clientes = [];
      try { clientes = await apiPost(ENDPOINTS.cliente, { nombre: q }); } catch(e){ console.error(e); }
      results.innerHTML = '';
      renderClientes(results, clientes);
      return;
    }

    // 4) Cualquier otra mezcla rara → intenta como RFC (con dígitos) o avisa
    results.appendChild(el('div',{class:'alert alert-warning'},'Formato no reconocido. Escribe solo números (IdOrden), RFC (con dígitos) o nombre (solo letras).'));
  };

  // ---- Modal (auto-inyección si no existe) ----
  const ensureModal = () => {
    if ($('#modal-reimpresion')) return;
    const html = `
<div class="modal fade" id="modal-reimpresion" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content rounded-4">
      <div class="modal-header">
        <h5 class="modal-title">Reimpresión de documentos</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label class="form-label">Buscar por RFC, Nombre de cliente o IdOrden</label>
          <div class="input-group">
            <input type="text" id="rei-input" class="form-control" placeholder="Ej.: ABCD001122XYZ / INDUSTRIAS / 2488" />
            <button class="btn btn-danger" id="rei-btn-buscar" type="button">Buscar</button>
          </div>
          <div class="form-text">Números = IdOrden. Con dígitos = RFC. Solo letras = nombre.</div>
        </div>
        <div id="rei-results"></div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-danger" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>`;
    const wrap = document.createElement('div');
    wrap.innerHTML = html;
    document.body.appendChild(wrap.firstElementChild);
  };

  // ---- API pública ----
  window.openReprintModal = () => {
    ensureModal();
    const m = new bootstrap.Modal($('#modal-reimpresion'));
    $('#rei-input').value = '';
    $('#rei-results').innerHTML = '';
    m.show();
    setTimeout(()=> $('#rei-input').focus(), 200);
  };

  // ---- Init ----
  document.addEventListener('DOMContentLoaded', () => {
    ensureModal();
    const btn = $('#rei-btn-buscar');
    const inp = $('#rei-input');
    if (btn) btn.addEventListener('click', doSearch);
    if (inp) inp.addEventListener('keydown', (e)=> { if (e.key==='Enter') { e.preventDefault(); doSearch(); } });
  });
})();
