// =============================================
//   DASHBOARD — GRÁFICOS MÚLTIPLES (3 meses)
// =============================================

document.addEventListener('DOMContentLoaded', function () {
    loadDashboardGraficos();
});

// Traducción tipo → nombre amigable
const TIPO_NOMBRE = {
    1: "Equipo Nuevo",
    2: "Recargas y Mantenimiento",
    3: "Capacitaciones",
    6: "PIPC",
    7: "Proyectos Especiales"
};

const DASHBOARD_ENDPOINT =
    "https://grupoxtinfire.com/admin/ajax/dashboard_ventas_read";


// Últimos 3 meses
function getUltimosTresMeses() {
    const hoy = new Date();
    const fin = hoy.toISOString().split("T")[0];

    const inicio = new Date(hoy.getFullYear(), hoy.getMonth() - 2, 1)
        .toISOString().split("T")[0];

    return { inicio, fin };
}


function loadDashboardGraficos() {
    const cont = document.getElementById("dashboard-graficos");
    if (!cont) return;

    const fechas = getUltimosTresMeses();

    let formData = new FormData();
    formData.append("hook", "codi_hook");
    formData.append("action", "load");
    formData.append("fechaInicio", fechas.inicio);
    formData.append("fechaFin", fechas.fin);

    // filtros globales
    if (window.dashboardFiltros) {
        formData.append("vendedor", window.dashboardFiltros.vendedor);
        formData.append("tipo", window.dashboardFiltros.tipo);
        formData.append("fechaInicio", window.dashboardFiltros.fechaInicio + " 00:00:00");
        formData.append("fechaFin", window.dashboardFiltros.fechaFin + " 23:59:59");
    }

    fetch(DASHBOARD_ENDPOINT, {
        method: "POST",
        body: formData
    })
        .then(res => res.json())
        .then(json => {
            const rows = json.data || json.rows || json.result || json;
            renderGraficos(rows);
        })
        .catch(err => {
            cont.innerHTML = `<p class="text-danger">Error cargando datos gráficos.</p>`;
            console.error(err);
        });
}


function renderGraficos(rows) {
    const cont = document.getElementById("dashboard-graficos");

    // UI HTML
    cont.innerHTML = `
    <div class="row g-4">

        <div class="col-12 col-md-6 position-relative">
            <button class="btn btn-sm btn-outline-secondary btn-zoom-chart" data-target="chartOrdenes">
                🔍 Ver en grande
            </button>
            <canvas id="chartOrdenes"></canvas>
        </div>

        <div class="col-12 col-md-6 position-relative">
            <button class="btn btn-sm btn-outline-secondary btn-zoom-chart" data-target="chartDinero">
                🔍 Ver en grande
            </button>
            <canvas id="chartDinero"></canvas>
        </div>

    </div>

    <div class="row mt-4" id="cards-resumen"></div>

    <div class="modal fade" id="chartModalFullscreen">
        <div class="modal-dialog modal-fullscreen">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Vista ampliada</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <canvas id="chartZoomCanvas"></canvas>
                </div>
            </div>
        </div>
    </div>
    `;

    // Agrupar por vendedor
    const vendedores = {};

    rows.forEach(r => {
        if (!vendedores[r.IdUsuario]) {
            vendedores[r.IdUsuario] = {
                nombre: r.vendedor,
                dinero: 0,
                ordenes: 0,
                porTipo: {}
            };
        }

        vendedores[r.IdUsuario].dinero += Number(r.total_neto_tipo);
        vendedores[r.IdUsuario].ordenes += Number(r.total_ordenes_tipo);

        vendedores[r.IdUsuario].porTipo[r.TipoOrden] =
            (vendedores[r.IdUsuario].porTipo[r.TipoOrden] || 0) +
            Number(r.total_ordenes_tipo);
    });

    const vendorNames = Object.values(vendedores).map(v => v.nombre);

    const tiposUnicos = [...new Set(rows.map(r => r.TipoOrden))];

    // Barras por tipo
    const datasetsOrdenes = tiposUnicos.map(tipo => ({
        label: TIPO_NOMBRE[tipo] || tipo,
        data: Object.values(vendedores).map(v => v.porTipo[tipo] || 0),
        borderWidth: 1
    }));

    // Plugin tooltip legend
    const legendTooltipPlugin = {
        id: 'legendTooltip',
        afterRender(chart) {
            chart.legend.legendItems.forEach(i => {
                const tipo = i.text;
                const entrada = Object.entries(TIPO_NOMBRE).find(e => e[1] === tipo);
                i.tooltip = entrada ? entrada[1] : tipo;
            });
        }
    };

    // ============================
    //    Gráfico ÓRDENES
    // ============================
    const canvas1 = document.getElementById("chartOrdenes");
    canvas1.parentElement.style.height = "350px";

    const chart1 = new Chart(canvas1.getContext("2d"), {
        type: "bar",
        data: {
            labels: vendorNames,
            datasets: datasetsOrdenes
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: "top",
                    labels: {
                        padding: 10,
                        usePointStyle: true,
                        boxWidth: 10
                    }
                }
            }
        },
        plugins: [legendTooltipPlugin]
    });

    canvas1.chart = chart1;


    // ============================
    //    Gráfico DINERO
    // ============================
    const canvas2 = document.getElementById("chartDinero");
    canvas2.parentElement.style.height = "350px";

    const chart2 = new Chart(canvas2.getContext("2d"), {
        type: "line",
        data: {
            labels: vendorNames,
            datasets: [{
                label: "Total Dinero",
                data: Object.values(vendedores).map(v => v.dinero),
                borderWidth: 2,
                tension: 0.3
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false
        },
        plugins: [legendTooltipPlugin]
    });

    canvas2.chart = chart2;


    // ============================
    //    ZOOM FULLSCREEN
    // ============================
    document.querySelectorAll(".btn-zoom-chart").forEach(btn => {
        btn.addEventListener("click", function () {
            const targetId = this.dataset.target;
            const original = document.getElementById(targetId);

            const modal = new bootstrap.Modal(document.getElementById("chartModalFullscreen"));
            modal.show();

            if (window.chartZoomInstance)
                window.chartZoomInstance.destroy();

            const ctx = document.getElementById("chartZoomCanvas").getContext("2d");

            window.chartZoomInstance = new Chart(ctx, {
                type: original.chart.config.type,
                data: original.chart.config.data,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { position: "bottom" } }
                }
            });
        });
    });


    // ============================
    //    CARDS RESUMEN
    // ============================

    function formatMoney(num) {
        return "$" + Number(num).toLocaleString("en-US", {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }

    const cards = document.getElementById("cards-resumen");
    Object.values(vendedores).forEach(v => {
        cards.innerHTML += `
        <div class="col-12 col-md-3">
            <div class="card border shadow-sm">
                <div class="card-body">
                    <h6 class="fw-bold">${v.nombre}</h6>
                    <p class="mb-1">Órdenes: <strong>${v.ordenes}</strong></p>
                    <p class="mb-1">Total dinero: <strong>${formatMoney(v.dinero)}</strong></p>
                    <p class="text-muted small">
                        Promedio mensual: ${formatMoney(v.dinero / 3)}
                    </p>
                </div>
            </div>
        </div>
        `;
    });
}
