// ============================================================
//   DASHBOARD — RANKING DE VENDEDORES
// ============================================================

document.addEventListener("DOMContentLoaded", () => {
    loadRankingVendedores();
});


/**
 * Carga datos del ranking desde el backend
 */
function loadRankingVendedores() {

    let formData = new FormData();
    formData.append("hook", "codi_hook");
    formData.append("action", "load");

    // Filtros globales (si existen)
    if (window.dashboardFiltros) {
        formData.append("vendedor", window.dashboardFiltros.vendedor);
        formData.append("tipo", window.dashboardFiltros.tipo);
        formData.append("fechaInicio", window.dashboardFiltros.fechaInicio + " 00:00:00");
        formData.append("fechaFin", window.dashboardFiltros.fechaFin + " 23:59:59");
    }

    fetch("https://grupoxtinfire.com/admin/ajax/dashboard_ranking_read", {
        method: "POST",
        body: formData
    })
        .then(r => r.json())
        .then(json => {
            const data = json.data || json.rows || json;

            renderRankingTable(data);
            renderRankingPodium(data);
            renderRankingChart(data);
        }).then(() => dashboardModuleLoaded())
        .catch(err => console.error("Error loading ranking:", err));
}


/* =======================================
   FORMATEADORES
======================================= */

function formatMoney(num) {
    return "$" + Number(num).toLocaleString("en-US", {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}


/* =======================================
   TABLA COMPLETA
======================================= */

function renderRankingTable(data) {
    const tbody = document.querySelector("#tablaRanking tbody");
    tbody.innerHTML = "";

    let pos = 1;

    data.forEach(r => {
        tbody.innerHTML += `
            <tr>
                <td>${pos}</td>
                <td>${r.Vendedor}</td>
                <td>${r.TotalOrdenes}</td>
                <td>${formatMoney(r.TotalDinero)}</td>
            </tr>
        `;
        pos++;
    });
}


/* =======================================
   PODIUM TOP 3
======================================= */

function renderRankingPodium(data) {

    const podium = document.getElementById("ranking-podium");
    podium.innerHTML = "";

    const colors = ["#ffd700", "#c0c0c0", "#cd7f32"]; // oro, plata, bronce

    data.slice(0, 3).forEach((r, i) => {
        podium.innerHTML += `
            <div class="col-12 col-md-4 text-center">
                <div class="p-3" 
                     style="border-radius:14px;
                            background:${colors[i]};
                            color:#000;
                            font-weight:600;">
                    
                    <h4 class="mb-1">${i + 1}° Lugar</h4>
                    <h5 class="fw-bold mb-1">${r.Vendedor}</h5>
                    <p class="mb-0">Órdenes: ${r.TotalOrdenes}</p>
                    <p class="mb-0">Ventas: ${formatMoney(r.TotalDinero)}</p>
                </div>
            </div>
        `;
    });
}


/* =======================================
   GRÁFICO — TOP VENDEDORES
======================================= */

function renderRankingChart(data) {

    const labels = data.map(r => r.Vendedor);
    const valores = data.map(r => r.TotalDinero);

    // Paleta XTINFIRE (10 colores)
    const PALETA = [
        "#dc3545", // rojo
        "#0d6efd", // azul
        "#198754", // verde
        "#ffc107", // amarillo
        "#6610f2", // morado
        "#fd7e14", // naranja
        "#20c997", // verde aqua
        "#6f42c1", // púrpura suave
        "#0dcaf0", // cian
        "#adb5bd"  // gris
    ];

    // Asignar colores cíclicamente
    const colores = labels.map((_, i) => PALETA[i % PALETA.length]);

    // Control del height del contenedor
    const canvas = document.getElementById("chartRanking");
    if (canvas) {
        canvas.parentElement.style.height = "350px";
    }

    // Render con ChartManager
    ChartManager.render("chartRanking", {
        type: "bar",
        data: {
            labels: labels,
            datasets: [
                {
                    label: "Ventas (dinero)",
                    data: valores,
                    backgroundColor: colores,
                    borderRadius: 8
                }
            ]
        },
        options: {
            indexAxis: "y",
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: {
                    beginAtZero: true,
                    ticks: {
                        callback: (v) => "$" + Number(v).toLocaleString("en-US")
                    }
                }
            },
            plugins: {
                legend: { display: false },
                tooltip: {
                    callbacks: {
                        label: (ctx) =>
                            " $" + Number(ctx.raw)
                                .toLocaleString("en-US", { minimumFractionDigits: 2 })
                    }
                }
            }
        }
    });
}

