// =============================================
//   DASHBOARD — STACKED POR TIPO / VENDEDOR
// =============================================

document.addEventListener("DOMContentLoaded", function () {
    loadStackedChart();
});

function loadStackedChart() {

    let formData = new FormData();
    formData.append("hook", "codi_hook");
    formData.append("action", "load");

    if (window.dashboardFiltros) {
        formData.append("vendedor", window.dashboardFiltros.vendedor);
        formData.append("tipo", window.dashboardFiltros.tipo);
        formData.append("fechaInicio", window.dashboardFiltros.fechaInicio + " 00:00:00");
        formData.append("fechaFin", window.dashboardFiltros.fechaFin + " 23:59:59");
    }

    fetch("https://grupoxtinfire.com/admin/ajax/dashboard_stacked_read", {
        method: "POST",
        body: formData
    })
        .then(r => r.json())
        .then(json => {
            const rows = json.data;

            if (!Array.isArray(rows)) {
                console.error("❌ Stacked no recibió un array");
                return;
            }

            console.log("STACKED RAW:", rows);

            //====================================
            //   1. Agrupar por vendedor
            //====================================
            const vendedores = {};
            const tipos = new Set();

            rows.forEach(r => {
                tipos.add(r.TipoOrden);

                if (!vendedores[r.IdUsuario]) {
                    vendedores[r.IdUsuario] = {
                        nombre: r.Vendedor,
                        tipos: {}
                    };
                }

                vendedores[r.IdUsuario].tipos[r.TipoOrden] = Number(r.TotalOrdenes);
            });

            //====================================
            //   2. Crear labels de vendedores
            //====================================
            const labels = Object.values(vendedores).map(v => v.nombre);

            //====================================
            //   3. Crear datasets por TipoOrden
            //====================================
            const datasets = [];

            tipos.forEach(tipo => {
                datasets.push({
                    label: getTipoNombre(tipo),
                    data: Object.values(vendedores).map(v => v.tipos[tipo] || 0),
                    backgroundColor: getColorForTipo(tipo)
                });
            });

            renderStackedChart(labels, datasets);
        }).then(() => dashboardModuleLoaded())
        .catch(err => console.error("Error stacked:", err));
}



// =============================================
//   DIBUJAR CHART
// =============================================
function renderStackedChart(labels, datasets) {

    const canvas = document.getElementById("chartStacked");
    if (!canvas) return;

    canvas.parentElement.style.height = "350px";

    ChartManager.render("chartStacked", {
        type: "bar",
        data: {
            labels: labels,
            datasets: datasets
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: { stacked: true },
                y: { stacked: true, beginAtZero: true }
            },
            plugins: {
                legend: {
                    position: "top",
                    labels: { usePointStyle: true }
                }
            }
        }
    });
}



// =============================================
//   UTILIDADES
// =============================================

// Traducción TipoOrden → Nombre
function getTipoNombre(tipo) {
    const mapa = {
        0: "Sin tipo",
        1: "Equipo Nuevo",
        2: "Recargas y Mantenimiento",
        3: "Capacitaciones",
        6: "PIPC",
        7: "Proyectos Especiales"
    };
    return mapa[tipo] || ("Tipo " + tipo);
}


// Color consistente por TipoOrden
function getColorForTipo(tipo) {
    const colores = {
        0: "#AAAAAA",
        1: "#FF6384",
        2: "#36A2EB",
        3: "#FFCE56",
        6: "#4BC0C0",
        7: "#9966FF"
    };
    return colores[tipo] || "#888888";
}
