// assets/js/dashboard-ventas.js

document.addEventListener('DOMContentLoaded', function () {
    initDashboardVentas();
});

// Endpoint AJAX
const DASHBOARD_VENTAS_ENDPOINT =
    'https://grupoxtinfire.com/admin/ajax/dashboard_ventas_read';

// META por tipo (importe $) para pintar palomita verde / rojo
const META_TIPO = 64000;

function initDashboardVentas() {
    const container = document.getElementById('dasboard-card-body');
    if (!container) return;

    container.innerHTML = `
      <div class="dv-filters mb-3">
        <div class="row g-2 align-items-end">
          <div class="col-12 col-md-3">
            <label for="dv-fecha-inicio" class="form-label mb-0">Desde</label>
            <input type="date" id="dv-fecha-inicio" class="form-control">
          </div>
          <div class="col-12 col-md-3">
            <label for="dv-fecha-fin" class="form-label mb-0">Hasta</label>
            <input type="date" id="dv-fecha-fin" class="form-control">
          </div>
          <div class="col-12 col-md-3">
            <button id="dv-btn-filtrar" class="btn btn-danger w-100 mt-3 mt-md-0">
              Mostrar datos
            </button>
          </div>
        </div>
        <p class="text-muted small mt-2 mb-0">
          Si la etiqueta está en rojo indica que tu meta no ha sido completada,
          si está en verde tu meta ha sido completada.
        </p>
      </div>
      <div class="table-responsive">
        <table class="table table-hover align-middle mb-0" id="dv-tabla-ventas"></table>
      </div>
    `;

    const today = new Date();
    const firstDay = new Date(today.getFullYear(), today.getMonth(), 1);

    document.getElementById('dv-fecha-inicio').value = formatDateInput(firstDay);
    document.getElementById('dv-fecha-fin').value = formatDateInput(today);

    document
        .getElementById('dv-btn-filtrar')
        .addEventListener('click', fetchDashboardVentas);

    fetchDashboardVentas();
}

function formatDateInput(date) {
    const y = date.getFullYear();
    const m = String(date.getMonth() + 1).padStart(2, '0');
    const d = String(date.getDate()).padStart(2, '0');
    return `${y}-${m}-${d}`;
}

function fetchDashboardVentas() {
    const tabla = document.getElementById('dv-tabla-ventas');
    if (!tabla) return;

    const inicio = document.getElementById('dv-fecha-inicio').value;
    const fin = document.getElementById('dv-fecha-fin').value;

    tabla.innerHTML = `
      <tbody>
        <tr>
          <td class="text-center py-5" colspan="99">
            <div class="spinner-border" role="status"></div>
            <div class="mt-2 small text-muted">Cargando datos...</div>
          </td>
        </tr>
      </tbody>
    `;

    const formData = new FormData();
    formData.append('hook', 'codi_hook');
    formData.append('action', 'load');
    formData.append('fechaInicio', inicio);
    formData.append('fechaFin', fin);

    fetch(DASHBOARD_VENTAS_ENDPOINT, {
        method: 'POST',
        body: formData
    })
        .then(function (response) {
            return response.json();
        })
        .then(function (json) {
            const rows = json.data || json.rows || json.result || json;
            buildDashboardVentasTable(Array.isArray(rows) ? rows : []);
        })
        .catch(function (error) {
            console.error(error);
            tabla.innerHTML = `
              <tbody>
                <tr>
                  <td class="text-center text-danger py-4" colspan="99">
                    Ocurrió un error al cargar la información.
                  </td>
                </tr>
              </tbody>
            `;
        });
}

function buildDashboardVentasTable(rows) {
    const tabla = document.getElementById('dv-tabla-ventas');
    if (!tabla) return;

    if (!rows || rows.length === 0) {
        tabla.innerHTML = `
          <tbody>
            <tr>
              <td class="text-center py-4" colspan="99">
                No se encontraron registros para el rango de fechas seleccionado.
              </td>
            </tr>
          </tbody>
        `;
        return;
    }

    const vendedoresMap = {};
    const tiposList = [];
    const tiposIndex = {};

    rows.forEach(function (row) {
        const idUsu = row.IdUsuario;
        const vendedorNombre = row.vendedor || row.Usuario || 'Sin nombre';
        const tipoId = String(row.TipoOrden);
        const tipoNombre = row.tipo_orden || row.descripcion || 'Sin tipo';

        if (!vendedoresMap[idUsu]) {
            vendedoresMap[idUsu] = {
                IdUsuario: idUsu,
                vendedor: vendedorNombre,
                tipos: {},
                totalDinero: 0,
                totalOrdenes: 0
            };
        }

        if (tiposIndex[tipoId] === undefined) {
            tiposIndex[tipoId] = tiposList.length;
            tiposList.push({
                id: tipoId,
                nombre: tipoNombre
            });
        }

        const totalNetoTipo = Number(row.total_neto_tipo || 0);
        const totalOrdenesTipo = Number(row.total_ordenes_tipo || 0);

        vendedoresMap[idUsu].tipos[tipoId] = {
            nombre: tipoNombre,
            totalNeto: totalNetoTipo,
            totalOrdenes: totalOrdenesTipo
        };

        vendedoresMap[idUsu].totalDinero += totalNetoTipo;
        vendedoresMap[idUsu].totalOrdenes += totalOrdenesTipo;
    });

    const vendedores = Object.values(vendedoresMap).sort(function (a, b) {
        return a.vendedor.localeCompare(b.vendedor, 'es');
    });

    const globalTotals = {
        perTipoImporte: {},
        perTipoOrdenes: {},
        totalDinero: 0,
        totalOrdenes: 0
    };

    let theadHtml = `
      <thead class="table-light align-middle">
        <tr>
          <th class="text-nowrap">Integrantes del Grupo</th>
    `;

    tiposList.forEach(function (t) {
        theadHtml += `<th class="text-center text-nowrap">${escapeHtml(t.nombre)}</th>`;
    });

    theadHtml += `
          <th class="text-center text-nowrap">Total Dinero</th>
          <th class="text-center text-nowrap">Total Órdenes</th>
          <th class="text-center text-nowrap">Cantidad en Unidades equivalente</th>
        </tr>
      </thead>
    `;

    let tbodyHtml = '<tbody>';

    vendedores.forEach(function (v) {
        globalTotals.totalDinero += v.totalDinero;
        globalTotals.totalOrdenes += v.totalOrdenes;

        let rowHtml = `
          <tr>
            <td class="fw-semibold dv-vendedor-cell">
              ${escapeHtml(v.vendedor)}
            </td>
        `;

        const resumenTipos = [];

        tiposList.forEach(function (t) {
            const tipoData = v.tipos[t.id];
            const importeTipo = tipoData ? tipoData.totalNeto : 0;
            const ordenesTipo = tipoData ? tipoData.totalOrdenes : 0;

            globalTotals.perTipoImporte[t.id] =
                (globalTotals.perTipoImporte[t.id] || 0) + importeTipo;
            globalTotals.perTipoOrdenes[t.id] =
                (globalTotals.perTipoOrdenes[t.id] || 0) + ordenesTipo;

            resumenTipos.push(`${formatInt(ordenesTipo)} ${t.nombre}`);

            // META: palomita verde si importeTipo >= META_TIPO
            // Calcular porcentaje
              let porcentaje = 0;
              if (META_TIPO > 0) {
                  porcentaje = (importeTipo / META_TIPO) * 100;
              }

              // Determinar icono (palomita si llegó al 100)
              const icon = porcentaje >= 100 ? 'check_circle' : 'error_outline';

              // Color SOLO DEL ÍCONO (NO del fondo)
              let iconColor = '';
              if (porcentaje < 90) {
                  iconColor = 'dv-icon-danger';     // rojo
              } else if (porcentaje >= 90 && porcentaje < 100) {
                  iconColor = 'dv-icon-warning';    // naranja
              } else {
                  iconColor = 'dv-icon-success';    // verde
              }

              // Mantener el fondo actual (NO CAMBIA)
              const pillClass = importeTipo > 0 ? 'dv-pill-success' : 'dv-pill-danger';

              rowHtml += `
                <td class="text-center">
                  <span class="dv-pill ${pillClass}">
                    <span class="material-icons-outlined dv-icon ${iconColor}">
                      ${icon}
                    </span>
                    <span>${formatMoney(importeTipo)}</span>
                  </span>
                </td>
              `;

        });

        rowHtml += `
            <td class="text-center fw-semibold">
              ${formatMoney(v.totalDinero)}
            </td>
            <td class="text-center fw-semibold">
              ${formatInt(v.totalOrdenes)}
            </td>
            <td class="text-start dv-resumen-unidades">
              ${escapeHtml(resumenTipos.join(', '))}.
            </td>
          </tr>
        `;

        tbodyHtml += rowHtml;
    });

    const resumenGlobal = tiposList
        .map(function (t) {
            const ords = globalTotals.perTipoOrdenes[t.id] || 0;
            return `${formatInt(ords)} ${t.nombre}`;
        })
        .join(', ');

    tbodyHtml += `
      <tr class="fw-semibold table-light">
        <td>SUMA TOTALES</td>
    `;

    tiposList.forEach(function (t) {
        const totalTipo = globalTotals.perTipoImporte[t.id] || 0;
        tbodyHtml += `
          <td class="text-center">
            ${formatMoney(totalTipo)}
          </td>
        `;
    });

    tbodyHtml += `
        <td class="text-center">
          ${formatMoney(globalTotals.totalDinero)}
        </td>
        <td class="text-center">
          ${formatInt(globalTotals.totalOrdenes)}
        </td>
        <td class="text-start">
          ${escapeHtml(resumenGlobal)}.
        </td>
      </tr>
      </tbody>
    `;

    tabla.innerHTML = theadHtml + tbodyHtml;
}

function formatMoney(value) {
    const num = Number(value) || 0;
    return '$' + num.toLocaleString('es-MX', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

function formatInt(value) {
    const num = Number(value) || 0;
    return num.toLocaleString('es-MX');
}

function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    return String(text)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}


