/* kpiordenes.js  (Órdenes 3M + Recall Recarga/Mantenimiento)
   - Render en #kpi-orders-card-body
   - Tablas con paginación 50/100/Todos, buscador multi-palabra, export XLSX/XLS
   - Gráficas (Chart.js)
   - Modales de detalle (tránsito, estatus y detalle de orden)
*/

(() => {
  const API_BASE = 'https://grupoxtinfire.com/admin/ajax/';
  const HOOK = 'codi_hook';
  const ACTION = 'load';
  const ROOT = '#kpi-orders-card-body';

  const ENDPOINTS = {
    // core
    kpi:                 'kpi_ordenes_3m',
    tendencia:           'tendencia_ordenes_mensual_3m',
    top10Clientes:       'top10_clientes_3m',
    totalesTransito:     'totales_transito_3m',
    detalleTransito:     'detalle_transito_3m',     // param: depto_id (int)
    totalesEstatus:      'totales_estatus_orden_3m',
    detalleEstatus:      'detalle_estatus_orden_3m',// param: estatus_id (int)
    porVendedor:         'ordenes_por_vendedor_3m',
    vendedorCategoria:   'ordenes_vendedor_categoria_3m',
    porTipoOrden:        'ordenes_por_tipo_orden_3m',

    // recall recarga/mantenimiento
    recargaRecall:       'ordenes_recarga_mantenimiento_recall',
    ordenDetalle:        'detalle_orden_by_id'      // param: id_orden (int)
  };

  // URL donde ver una orden en tu sistema
  const ORDER_URL_BASE = '/admin/ordenes/detalle.php?id=';

  const $ = (sel, ctx=document) => ctx.querySelector(sel);

  // ---- Helpers ----
  const el = (tag, attrs, ...children) => {
    const node = document.createElement(tag);
    const a = (attrs && typeof attrs === 'object') ? attrs : {};
    Object.entries(a).forEach(([k,v])=>{
      if (v === null || v === undefined || v === false) return;
      if (k === 'class') node.className = v;
      else if (k === 'dataset' && v && typeof v === 'object') Object.assign(node.dataset, v);
      else if (k.startsWith('on') && typeof v === 'function') node.addEventListener(k.slice(2), v);
      else node.setAttribute(k, v);
    });
    const flat = children.flat ? children.flat() : [].concat(...children);
    flat.forEach(c=>{
      if (c === null || c === undefined || c === false) return;
      if (typeof c === 'string' || typeof c === 'number') node.appendChild(document.createTextNode(String(c)));
      else if (c && typeof c === 'object' && 'nodeType' in c) node.appendChild(c);
      else node.appendChild(document.createTextNode(String(c)));
    });
    return node;
  };

  const currency = new Intl.NumberFormat('es-MX',{style:'currency',currency:'MXN'});
  const num = (n,d=0)=> (Number(n)||0).toLocaleString('es-MX',{minimumFractionDigits:d,maximumFractionDigits:d});
  const normalize = s => (s||'').toString().toLowerCase().normalize('NFD').replace(/\p{Diacritic}/gu,'');

  const loadScriptIfMissing = (globalName, src) => new Promise((resolve,reject)=>{
    if (window[globalName]) return resolve();
    const s = el('script',{src,async:true});
    s.onload = () => resolve();
    s.onerror = () => reject(new Error('No se pudo cargar: '+src));
    document.head.appendChild(s);
  });

  // XLSX con 3 CDNs en cascada
  const ensureXLSX = () => new Promise((resolve,reject)=>{
    if (window.XLSX) return resolve();
    const urls = [
      'https://cdn.sheetjs.com/xlsx-0.20.3/package/dist/xlsx.full.min.js',
      'https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js',
      'https://unpkg.com/xlsx@0.18.5/dist/xlsx.full.min.js'
    ];
    (function tryNext(i){
      if (i >= urls.length) return reject(new Error('No se pudo cargar XLSX'));
      const s = el('script',{src:urls[i],async:true});
      s.onload = () => resolve();
      s.onerror = () => tryNext(i+1);
      document.head.appendChild(s);
    })(0);
  });

  const apiPost = async (fnName, params={}) => {
    const body = new URLSearchParams({hook:HOOK, action:ACTION, ...params});
    const res = await fetch(API_BASE + fnName, {
      method:'POST',
      headers:{'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8'},
      body
    });
    if (!res.ok) throw new Error('HTTP '+res.status+' @'+fnName);
    const json = await res.json();
    if (!json || json.status !== 200 || json.error) throw new Error('API '+fnName+' error');
    return json.data || [];
  };

  // Export
  const tableDataForExport = (thead, data, renderRow) => {
    const headers = [...thead.querySelectorAll('th')].map(th => th.textContent.trim());
    const rows = data.map(r => {
      const tr = renderRow(r, true);
      return [...tr.children].map(td => td.textContent.trim());
    });
    return [headers, ...rows];
  };
  const downloadXLSX = async (rows, filename='reporte.xlsx') => { await ensureXLSX(); const ws=XLSX.utils.aoa_to_sheet(rows); const wb=XLSX.utils.book_new(); XLSX.utils.book_append_sheet(wb, ws, 'Datos'); XLSX.writeFile(wb, filename); };
  const downloadXLS  = (rows, filename='reporte.xls') => {
    const html = `<html><head><meta charset="utf-8"></head><body><table border="1">${rows.map(r=>'<tr>'+r.map(c=>`<td>${String(c).replace(/</g,'&lt;')}</td>`).join('')+'</tr>').join('')}</table></body></html>`;
    const blob = new Blob([html], {type:'application/vnd.ms-excel'}); const a=el('a',{download:filename,href:URL.createObjectURL(blob)}); document.body.appendChild(a); a.click(); URL.revokeObjectURL(a.href); a.remove();
  };

  // UI comunes
  const buildSpinner = root => root.appendChild(el('div',{class:'d-flex justify-content-center my-3'}, el('div',{class:'spinner-border text-secondary me-2'}),'Cargando...'));
  const buildCard = (title, content, right=null) =>
    el('div',{class:'card mt-3'},
      el('div',{class:'card-body'},
        el('div',{class:'d-flex justify-content-between align-items-center mb-2'},
          el('h6',{class:'m-0'},title),
          right || el('div',{})
        ),
        content
      )
    );

  // Tabla con paginación, buscador y export
  const makeTableUI = ({ title, columns, data, renderRow, extraLeftControls=[] }) => {
    const search = el('input',{type:'text',class:'form-control',placeholder:'Buscar (multi-palabra)'});
    const pageSel = el('select',{class:'form-select'},
      el('option',{value:'50'},'50'),
      el('option',{value:'100'},'100'),
      el('option',{value:'all'},'Todos')
    );
    const btnXLSX = el('button',{type:'button',class:'btn btn-sm btn-outline-primary'},'XLSX');
    const btnXLS  = el('button',{type:'button',class:'btn btn-sm btn-outline-secondary'},'XLS');

    const left = el('div',{class:'d-flex flex-wrap gap-2 align-items-center'}, ...extraLeftControls,
      el('div',{class:'d-flex align-items-center gap-2'}, el('label',{class:'small text-muted'},'Mostrar:'), pageSel)
    );
    const right = el('div',{class:'d-flex flex-wrap gap-2 align-items-center'}, search, btnXLSX, btnXLS);

    const table = el('table',{class:'table align-middle'},
      el('thead',{}, el('tr',{}, ...columns.map(h=>el('th',{},h)))),
      el('tbody',{})
    );
    const info = el('div',{class:'text-muted small'});
    const prev = el('button',{type:'button',class:'btn btn-sm btn-outline-secondary'},'Anterior');
    const next = el('button',{type:'button',class:'btn btn-sm btn-outline-secondary'},'Siguiente');

    const card = buildCard(title, el('div',{},
      el('div',{class:'d-flex justify-content-between gap-2 mb-2'}, left, right),
      el('div',{class:'table-responsive'}, table),
      el('div',{class:'d-flex justify-content-between align-items-center'}, info, el('div',{class:'d-flex gap-2'}, prev, next))
    ));

    let pageSize = 50, pageIndex = 0, tokens=[];
    let predicate = null;

    const normalizeRow = r => normalize(JSON.stringify(r));
    const getFiltered = () => (data||[]).filter(r => {
      const okSearch = tokens.every(t => normalizeRow(r).includes(t));
      const okPred = predicate ? predicate(r) : true;
      return okSearch && okPred;
    });

    const render = () => {
      const list = getFiltered();
      const total = list.length;
      const ps = pageSize===Infinity ? total : pageSize;
      const pages = ps ? Math.ceil(total/ps) : 1;
      if (pageIndex >= pages) pageIndex = 0;

      table.tBodies[0].innerHTML='';
      const start = pageIndex * (ps||1);
      const end   = pageSize===Infinity ? total : Math.min(start+ps, total);
      const slice = pageSize===Infinity ? list : list.slice(start,end);
      slice.forEach(row=> table.tBodies[0].appendChild(renderRow(row,false)));

      const from = total ? (start+1) : 0;
      info.textContent = `Mostrando ${from}-${end} de ${total}`;
      prev.disabled = (pageIndex<=0); next.disabled=(pageIndex>=pages-1 || pages<=1);
    };

    search.addEventListener('input', ()=>{ tokens=normalize(search.value).split(/\s+/).filter(Boolean); pageIndex=0; render(); });
    pageSel.addEventListener('change', ()=>{ const v=pageSel.value; pageSize=(v==='all')?Infinity:parseInt(v,10); pageIndex=0; render(); });
    prev.addEventListener('click', ()=>{ if(pageIndex>0){ pageIndex--; render(); }});
    next.addEventListener('click', ()=>{ pageIndex++; render(); });

    btnXLSX.addEventListener('click', async ()=>{
      const rows = tableDataForExport(table.tHead, getFiltered(), renderRow);
      await downloadXLSX(rows, (title||'reporte')+'.xlsx');
    });
    btnXLS.addEventListener('click', ()=>{
      const rows = tableDataForExport(table.tHead, getFiltered(), renderRow);
      downloadXLS(rows, (title||'reporte')+'.xls');
    });

    return { card, table, render, setPredicate:(fn)=>{predicate=fn; render();}, setPageSize:(v)=>{pageSize=v; render();} };
  };

  // Modal genérico
  const ensureModal = (id, title='Detalle') => {
    let modal = document.getElementById(id);
    if (!modal) {
      modal = el('div',{class:'modal fade', id, tabindex:'-1'},
        el('div',{class:'modal-dialog modal-xl modal-dialog-scrollable'},
          el('div',{class:'modal-content'},
            el('div',{class:'modal-header'},
              el('h5',{class:'modal-title'}, title),
              el('button',{type:'button',class:'btn-close','data-bs-dismiss':'modal','aria-label':'Close'})
            ),
            el('div',{class:'modal-body', id: id+'-body'}, 'Cargando...'),
            el('div',{class:'modal-footer'},
              el('button',{type:'button',class:'btn btn-secondary','data-bs-dismiss':'modal'},'Cerrar')
            )
          )
        )
      );
      document.body.appendChild(modal);
    }
    return modal;
  };

  // ---- Secciones ----
  const buildKpis = (root, k) => {
    const row = el('div',{class:'row g-3'},
      el('div',{class:'col-12 col-md-3'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Órdenes (3M)'),
          el('div',{class:'fw-bold fs-5'}, num(k.ordenes_3m||0))
        )
      ),
      el('div',{class:'col-12 col-md-3'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Clientes (3M)'),
          el('div',{class:'fw-bold fs-5'}, num(k.clientes_3m||0))
        )
      ),
      el('div',{class:'col-12 col-md-3'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Ingresos (3M)'),
          el('div',{class:'fw-bold fs-5'}, currency.format(k.ingresos_3m||0))
        )
      ),
      el('div',{class:'col-12 col-md-3'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Ticket promedio'),
          el('div',{class:'fw-bold fs-5'}, currency.format(k.ticket_promedio_3m||0))
        )
      )
    );
    root.appendChild(row);
  };

  const buildTendencia = (root, rows) => {
    const canvas = el('canvas',{id:'chartOrdenes',height:'110'});
    root.appendChild(buildCard('Tendencia mensual (Órdenes)', canvas));
    const labels  = (rows||[]).map(r=>r.mes_label||r.mes_key);
    const ords    = (rows||[]).map(r=>+r.ordenes||0);
    const ingresos= (rows||[]).map(r=>+r.ingresos||0);
    new Chart(canvas.getContext('2d'),{
      type:'line',
      data:{ labels, datasets:[
        {label:'Órdenes', data:ords, yAxisID:'y1', tension:.3},
        {label:'Ingresos', data:ingresos, yAxisID:'y2', tension:.3}
      ]},
      options:{responsive:true, scales:{ y1:{type:'linear',position:'left',beginAtZero:true}, y2:{type:'linear',position:'right',beginAtZero:true} }}
    });
  };

  const buildTopClientes = (root, rows) => {
    const ui = makeTableUI({
      title:'Top 10 clientes por ingresos (3M)',
      columns:['Cliente','Órdenes','Ingresos'],
      data: rows,
      renderRow: r => el('tr',{},
        el('td',{}, r.cliente||''),
        el('td',{}, num(r.ordenes||0)),
        el('td',{}, currency.format(r.ingresos||0))
      )
    });
    ui.table.setAttribute('data-table','top-clientes');
    root.appendChild(ui.card); ui.render();
  };

  const buildTransitoBoxes = (root, rows) => {
    const sel = el('select',{class:'form-select w-auto'}, el('option',{value:''},'Selecciona tránsito'));
    rows.forEach(r=> sel.appendChild(el('option',{value:r.IdDepto}, `${r.transito} (ID ${r.IdDepto})`)));
    const total = rows.reduce((a,b)=>a+(+b.ordenes||0),0);
    const btn = el('button',{type:'button',class:'btn btn-sm btn-primary'},'Ver órdenes');

    const box = el('div',{class:'d-flex align-items-center gap-3'},
      el('div',{}, el('div',{class:'small text-muted'},'Total (3M)'), el('div',{class:'fw-bold fs-4'}, num(total))),
      el('div',{}, el('label',{class:'small text-muted'},'Estado de tránsito'), sel),
      btn
    );
    root.appendChild(buildCard('Órdenes por Estado de Tránsito (3M)', box));

    const modal = ensureModal('modal-transito','Órdenes por Tránsito');

    btn.addEventListener('click', async ()=>{
      const val = sel.value; if (!val) return;
      const data = await apiPost(ENDPOINTS.detalleTransito, {depto_id: parseInt(val,10)||0});
      const ui = makeTableUI({
        title: `Detalle — Tránsito ID ${val}`,
        columns:['IdOrden','Fecha','Cliente','Estatus','Tipo de Orden','Vendedor','Total'],
        data,
        renderRow: r => el('tr',{},
          el('td',{}, r.IdOrden),
          el('td',{}, r.Fecha || ''),
          el('td',{}, r.cliente || ''),
          el('td',{}, r.estatus_orden || ''),
          el('td',{}, r.tipo_orden || ''),
          el('td',{}, r.vendedor || ''),
          el('td',{}, currency.format(r.total||0))
        )
      });
      const body = document.getElementById('modal-transito-body'); body.innerHTML=''; body.appendChild(ui.card); ui.render();
      new bootstrap.Modal(modal).show();
    });
  };

  const buildEstatusBoxes = (root, rows) => {
    const sel = el('select',{class:'form-select w-auto'}, el('option',{value:''},'Selecciona estatus'));
    rows.forEach(r=> sel.appendChild(el('option',{value:r.IdEstatus}, `${r.estatus} (ID ${r.IdEstatus})`)));
    const total = rows.reduce((a,b)=>a+(+b.ordenes||0),0);
    const btn = el('button',{type:'button',class:'btn btn-sm btn-primary'},'Ver órdenes');

    const box = el('div',{class:'d-flex align-items-center gap-3'},
      el('div',{}, el('div',{class:'small text-muted'},'Total (3M)'), el('div',{class:'fw-bold fs-4'}, num(total))),
      el('div',{}, el('label',{class:'small text-muted'},'Estatus de orden'), sel),
      btn
    );
    root.appendChild(buildCard('Órdenes por Estatus (3M)', box));

    const modal = ensureModal('modal-estatus','Órdenes por Estatus');

    btn.addEventListener('click', async ()=>{
      const val = sel.value; if (!val) return;
      const data = await apiPost(ENDPOINTS.detalleEstatus, {estatus_id: parseInt(val,10)||0});
      const ui = makeTableUI({
        title: `Detalle — Estatus ID ${val}`,
        columns:['IdOrden','Fecha','Cliente','Tránsito','Tipo de Orden','Vendedor','Total'],
        data,
        renderRow: r => el('tr',{},
          el('td',{}, r.IdOrden),
          el('td',{}, r.Fecha || ''),
          el('td',{}, r.cliente || ''),
          el('td',{}, r.transito || ''),
          el('td',{}, r.tipo_orden || ''),
          el('td',{}, r.vendedor || ''),
          el('td',{}, currency.format(r.total||0))
        )
      });
      const body = document.getElementById('modal-estatus-body'); body.innerHTML=''; body.appendChild(ui.card); ui.render();
      new bootstrap.Modal(modal).show();
    });
  };

  const buildVendedores = (root, rows) => {
    const ui = makeTableUI({
      title:'Órdenes por vendedor (3M)',
      columns:['Vendedor','Órdenes','Ingresos'],
      data: rows,
      renderRow: r => el('tr',{},
        el('td',{}, r.vendedor || ''),
        el('td',{}, num(r.ordenes||0)),
        el('td',{}, currency.format(r.ingresos||0))
      )
    });
    ui.table.setAttribute('data-table','vendedores');
    root.appendChild(ui.card); ui.render();
  };

  const buildVendedorCategoria = (root, rows) => {
    const sel = el('select',{class:'form-select w-auto'}, el('option',{value:''},'Todos los vendedores'));
    [...new Map((rows||[]).map(r=>[r.IdUsuario,r.vendedor])).entries()].forEach(([id,nom])=>{
      sel.appendChild(el('option',{value:id}, nom || `Usuario ${id}`));
    });

    const ui = makeTableUI({
      title:'Órdenes por vendedor y categoría (3M)',
      columns:['Vendedor','Categoría','Órdenes','Ingresos'],
      data: rows || [],
      renderRow: r => el('tr',{},
        el('td',{}, r.vendedor || ''),
        el('td',{}, r.categoria || '-'),
        el('td',{}, num(r.ordenes||0)),
        el('td',{}, currency.format(r.ingresos||0))
      ),
      extraLeftControls:[sel]
    });
    ui.setPredicate(row => !sel.value || String(row.IdUsuario) === String(sel.value));
    sel.addEventListener('change', ()=> ui.render());

    ui.table.setAttribute('data-table','vend-cat');
    root.appendChild(ui.card); ui.render();
  };

  const buildTipoOrden = (root, rows) => {
    const ui = makeTableUI({
      title:'Órdenes por tipo de orden (3M)',
      columns:['Tipo de Orden','Órdenes','Ingresos'],
      data: rows,
      renderRow: r => el('tr',{},
        el('td',{}, r.tipo_orden || '(sin tipo)'),
        el('td',{}, num(r.ordenes||0)),
        el('td',{}, currency.format(r.ingresos||0))
      )
    });
    root.appendChild(ui.card); ui.render();
  };

  // Recarga/Mantenimiento (recall) m-11 y m-10

const buildRecargaRecall = (root, rows) => {
  const modal = ensureModal('modal-orden-detalle', 'Detalle de Orden');
  const modalBody = document.getElementById('modal-orden-detalle-body');

  const verDetalle = async (idOrden) => {
    modalBody.innerHTML = 'Cargando...';
    try {
      const det = await apiPost(ENDPOINTS.ordenDetalle, { id_orden: parseInt(idOrden,10)||0 });
      const ui = makeTableUI({
        title: `Orden #${idOrden} — Productos`,
        columns: ['SKU','Producto','Cantidad','Precio','Neto'],
        data: det || [],
        renderRow: r => el('tr',{},
          el('td',{}, r.proSku || ''),
          el('td',{}, r.producto || ''),
          el('td',{}, (r.Cantidad ?? r.cantidad ?? 0)),
          el('td',{}, currency.format(r.PrecioVenta ?? r.precio ?? 0)),
          el('td',{}, currency.format(r.Neto ?? 0))
        )
      });
      modalBody.innerHTML = ''; modalBody.appendChild(ui.card); ui.render();
      new bootstrap.Modal(modal).show();
    } catch (e) {
      modalBody.innerHTML = `<div class="alert alert-danger">No se pudo cargar el detalle: ${e.message}</div>`;
      new bootstrap.Modal(modal).show();
    }
  };

  const ui = makeTableUI({
    title: 'Recall Recarga/Mantenimiento (tipo_orden = 2) — Meses m-11 y m-10',
    columns: ['Fecha Orden','Orden','Cliente','Teléfono','Correo','Vendedor','Ítems','Unidades','Total','Pedido actual (mes)'],
    data: rows || [],
    renderRow: r => {
      const linkOrden = el('a', { href: ORDER_URL_BASE + r.IdOrden, target: '_blank' }, '#' + r.IdOrden);
      const btnVer = el('button', { type:'button', class:'btn btn-sm btn-outline-primary ms-2', onclick: ()=>verDetalle(r.IdOrden) }, 'Ver');

      let pedidoActualCell = el('span', { class: 'text-muted' }, '—');
      if (r.orden_mes_actual) {
        const a = el('a', { href: ORDER_URL_BASE + r.orden_mes_actual, target: '_blank' }, '#' + r.orden_mes_actual);
        const btn = el('button', { type:'button', class:'btn btn-sm btn-outline-secondary ms-2', onclick: ()=>verDetalle(r.orden_mes_actual) }, 'Ver');
        pedidoActualCell = el('span', {}, a, btn);
      }

      return el('tr',{},
        el('td',{}, r.Fecha || ''),
        el('td',{}, linkOrden, btnVer),
        el('td',{}, r.cliente || ''),
        el('td',{}, r.telefono || ''),
        el('td',{}, r.email || ''),
        el('td',{}, r.vendedor || ''),             // ⬅️ nueva columna
        el('td',{}, (r.items ?? 0)),
        el('td',{}, (r.unidades ?? 0)),
        el('td',{}, currency.format(r.total ?? 0)),
        el('td',{}, pedidoActualCell)
      );
    }
  });

  ui.table.setAttribute('data-table','recarga-recall');
  root.appendChild(ui.card);
  ui.render();
};


  // PDF toolbar (gráfica + tablas principales)
  const buildPdfToolbar = (root) => {
    const form = el('form',{id:'pdfOrdenesForm',method:'post',action:'https://grupoxtinfire.com/admin/pdfreportes/kpi_ordenes_pdf',class:'d-flex align-items-center gap-2 mb-3'},
      el('input',{type:'hidden',name:'chart_ordenes',id:'chart_ordenes'}),
      el('input',{type:'hidden',name:'tabla_top_clientes',id:'tabla_top_clientes'}),
      el('input',{type:'hidden',name:'tabla_vendedor_cat',id:'tabla_vendedor_cat'}),
      el('input',{type:'hidden',name:'tabla_recarga_recall',id:'tabla_recarga_recall'}),
      el('input',{type:'hidden',name:'tabla_vendedores',id:'tabla_vendedores'}),
      el('button',{type:'button',id:'btnPdfOrdenes',class:'btn btn-sm btn-dark'},
        el('span',{class:'material-icons-outlined align-middle me-1'},'picture_as_pdf'),'Descargar PDF'
      )
    );
    root.appendChild(form);

    $('#btnPdfOrdenes').addEventListener('click', ()=>{
      try{
        const ch = document.getElementById('chartOrdenes');
        $('#chart_ordenes').value = ch ? ch.toDataURL('image/png') : '';

        const q = sel => {
          const t = document.querySelector(sel);
          return t ? t.outerHTML : '';
        };
        // marca data-table para capturar HTML exacto
        $('#tabla_top_clientes').value  = q('#kpi-orders-card-body table[data-table="top-clientes"]');
        $('#tabla_vendedor_cat').value  = q('#kpi-orders-card-body table[data-table="vend-cat"]');
        $('#tabla_recarga_recall').value= q('#kpi-orders-card-body table[data-table="recarga-recall"]');
        $('#tabla_vendedores').value    = q('#kpi-orders-card-body table[data-table="vendedores"]');

        $('#pdfOrdenesForm').submit();
      }catch(e){ alert('No se pudo preparar el PDF: '+e.message); }
    });
  };

  // ---- Main ----
  document.addEventListener('DOMContentLoaded', async ()=>{
    const root = $(ROOT); if (!root) return;
    buildPdfToolbar(root);
    const spinner = buildSpinner(root);

    try{
      await loadScriptIfMissing('Chart','https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js');

      const [
        kpi, tendencia, top10, transito, estatus, porVend, vendCat, tipoOrden, recargaRecall
      ] = await Promise.all([
        apiPost(ENDPOINTS.kpi),
        apiPost(ENDPOINTS.tendencia),
        apiPost(ENDPOINTS.top10Clientes),
        apiPost(ENDPOINTS.totalesTransito),
        apiPost(ENDPOINTS.totalesEstatus),
        apiPost(ENDPOINTS.porVendedor),
        apiPost(ENDPOINTS.vendedorCategoria),
        apiPost(ENDPOINTS.porTipoOrden),
        apiPost(ENDPOINTS.recargaRecall)
      ]);

      spinner.remove();

      buildKpis(root, (kpi && kpi[0]) || {});
      buildTendencia(root, tendencia || []);
      buildTopClientes(root, top10 || []);
      buildTransitoBoxes(root, transito || []);
      buildEstatusBoxes(root, estatus || []);
      buildVendedores(root, porVend || []);
      buildVendedorCategoria(root, vendCat || []);
      buildTipoOrden(root, tipoOrden || []);
      buildRecargaRecall(root, recargaRecall || []);

    }catch(e){
      console.error(e);
      spinner.replaceWith(el('div',{class:'alert alert-danger'},'No se pudieron cargar los reportes. ', el('small',{class:'text-muted d-block mt-1'}, e.message)));
    }
  });
})();
