/* kpiproductos.js (v2 estable)
   - Render dentro de #kpi-card-body
   - Chart.js, tablas con paginación (50/100/Todos), buscador multi-palabra,
     exportar a XLSX/XLS, y envío de gráficos + 2 tablas al PDF (TCPDF).
*/

(() => {
  // ===== Config =====
  const API_BASE = 'https://grupoxtinfire.com/admin/ajax/';
  const HOOK = 'codi_hook';
  const ACTION = 'load';
  const TARGET = '#kpi-card-body';

  const ENDPOINTS = {
    kpiGlobal:           'kpi_productos_3m',
    trendMensual:        'tendencia_mensual_3m',
    ventasAlmacen:       'ventas_almacen_3m_summary',
    tendenciaAlmacen:    'tendencia_almacen_3m',
    top10Sku:            'top10_sku_3m',
    sinVentas:           'sin_ventas_3m',
    promos:              'promos_activas_hoy',
    ventasSkuAlmacen:    'ventas_sku_almacen_3m',
    saludSkuGlobal:      'salud_inventario_sku',
    saludSkuAlmacen:     'salud_inventario_almacen',
    topVendidosStock:    'top_vendidos_stock_3m',
    unsoldStock:         'sin_ventas_stock_3m'
  };

  // ===== Estado =====
  const STATE = { charts: { general: null, almacenes: null }, data: {} };

  // ===== Utils =====
  const $ = (sel, ctx = document) => ctx.querySelector(sel);

  // Helper robusto para crear nodos
  const el = (tag, attrs = {}, ...children) => {
    const node = document.createElement(tag);
    const a = (attrs && typeof attrs === 'object') ? attrs : {};
    for (const [k, v] of Object.entries(a)) {
      if (k === 'class') node.className = v;
      else if (k === 'dataset' && v && typeof v === 'object') Object.assign(node.dataset, v);
      else if (k.startsWith('on') && typeof v === 'function') node.addEventListener(k.slice(2), v);
      else if (v !== null && v !== undefined && v !== false) node.setAttribute(k, v);
    }
    const flat = Array.isArray(children) ? children.flat() : children;
    flat.forEach(c => {
      if (c === null || c === undefined || c === false) return;
      if (typeof c === 'string' || typeof c === 'number') {
        node.appendChild(document.createTextNode(String(c)));
      } else if (c && typeof c === 'object' && 'nodeType' in c) {
        node.appendChild(c);
      } else {
        node.appendChild(document.createTextNode(String(c)));
      }
    });
    return node;
  };

  const currency = new Intl.NumberFormat('es-MX', { style: 'currency', currency: 'MXN', maximumFractionDigits: 2 });
  const num = (n, d = 0) => (Number(n) || 0).toLocaleString('es-MX', { minimumFractionDigits: d, maximumFractionDigits: d });
  const normalize = (s) => (s || '').toString().toLowerCase().normalize('NFD').replace(/\p{Diacritic}/gu, '');

  const loadScriptIfMissing = (globalName, src) => new Promise((resolve, reject) => {
    if (window[globalName]) return resolve();
    const s = el('script', { src, async: true });
    s.onload = () => resolve();
    s.onerror = () => reject(new Error('No se pudo cargar: ' + src));
    document.head.appendChild(s);
  });

  const apiPost = async (fnName, extra = {}) => {
    const body = new URLSearchParams({ hook: HOOK, action: ACTION, ...extra });
    const res = await fetch(API_BASE + fnName, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body
    });
    if (!res.ok) throw new Error('HTTP ' + res.status + ': ' + fnName);
    const json = await res.json();
    if (!json || json.status !== 200 || json.error) throw new Error('API error: ' + fnName);
    return json.data || [];
  };

  // ===== Export a Excel =====
  const ensureXLSX = () => loadAnyScript([
  // Opción recomendada: CDN oficial de SheetJS (elige una versión estable)
  'https://cdn.sheetjs.com/xlsx-0.20.3/package/dist/xlsx.full.min.js',
  // Fallbacks (versión 0.18.5, la "latest" pública en npm CDNs)
  'https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js',
  'https://unpkg.com/xlsx@0.18.5/dist/xlsx.full.min.js'
]);

function loadAnyScript(urls) {
  return new Promise((resolve, reject) => {
    if (window.XLSX) return resolve();
    (function tryNext(i) {
      if (i >= urls.length) return reject(new Error('No se pudo cargar XLSX desde ningún CDN'));
      const s = document.createElement('script');
      s.src = urls[i];
      s.async = true;
      s.onload = () => resolve();
      s.onerror = () => tryNext(i + 1);
      document.head.appendChild(s);
    })(0);
  });
}
  const downloadXLSX = async (rows, filename) => {
    await ensureXLSX();
    const ws = XLSX.utils.aoa_to_sheet(rows);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Datos');
    XLSX.writeFile(wb, filename || 'reporte.xlsx');
  };

  const downloadXLS = (rows, filename) => {
    const html = `
      <html><head><meta charset="utf-8"></head><body>
      <table border="1" cellspacing="0" cellpadding="2">
        ${rows.map(r => '<tr>' + r.map(c => `<td>${String(c).replace(/</g, '&lt;')}</td>`).join('') + '</tr>').join('')}
      </table>
      </body></html>`;
    const blob = new Blob([html], { type: 'application/vnd.ms-excel' });
    const a = el('a', { download: filename || 'reporte.xls', href: URL.createObjectURL(blob) });
    document.body.appendChild(a); a.click(); URL.revokeObjectURL(a.href); a.remove();
  };

  const tableDataForExport = (thead, data, renderRow) => {
    const headers = [...thead.querySelectorAll('th')].map(th => th.textContent.trim());
    const rows = data.map(item => {
      const tr = renderRow(item, true); // modo export
      const cells = [...tr.children].map(td => td.textContent.trim());
      return cells;
    });
    return [headers, ...rows];
  };

  // ===== Componentes comunes =====
  const buildToolbar = (root) => {
    const form = el('form', { id: 'pdfForm', method: 'post', action: 'https://grupoxtinfire.com/admin/pdfreportes/kpi_pdf', class: 'd-flex align-items-center gap-2 mb-3' },
      el('input', { type: 'hidden', name: 'chart_general', id: 'chart_general' }),
      el('input', { type: 'hidden', name: 'chart_almacenes', id: 'chart_almacenes' }),
      el('input', { type: 'hidden', name: 'tabla_top', id: 'tabla_top' }),
      el('input', { type: 'hidden', name: 'tabla_kits', id: 'tabla_kits' }),
      el('button', { type: 'button', id: 'btnDescargarPdf', class: 'btn btn-sm btn-dark' },
        el('span', { class: 'material-icons-outlined align-middle me-1' }, 'picture_as_pdf'),
        'Descargar PDF'
      )
    );
    root.appendChild(form);
  };

  const buildSpinner = (root) =>
    root.appendChild(el('div', { id: 'kpi-loading', class: 'd-flex align-items-center justify-content-center my-4' },
      el('div', { class: 'spinner-border text-secondary me-2' }), 'Cargando reportes...'));

  const cardKpi = (title, val, small) =>
    el('div', { class: 'col-12 col-md-3' },
      el('div', { class: 'p-3 bg-light rounded-3 h-100' },
        el('div', { class: 'text-muted small mb-1' }, title),
        el('div', { class: 'fw-bold fs-5' }, val),
        small ? el('div', { class: 'text-muted small' }, small) : ''
      )
    );

  const buildCard = (title, content, actionsRight) =>
    el('div', { class: 'card mt-3' },
      el('div', { class: 'card-body' },
        el('div', { class: 'd-flex justify-content-between align-items-center mb-2' },
          el('h6', { class: 'card-title m-0' }, title),
          actionsRight || el('div', {})
        ),
        content
      )
    );

  // ===== Tabla UI con paginación/búsqueda/export =====
  const makeTableUI = ({ title, columns, data, renderRow, extraLeftControls }) => {
    const search = el('input', { type: 'text', class: 'form-control', placeholder: 'Buscar (multi-palabra)' });
    const pageSel = el('select', { class: 'form-select' },
      el('option', { value: '50' }, '50'),
      el('option', { value: '100' }, '100'),
      el('option', { value: 'all' }, 'Todos')
    );
    const btnXLSX = el('button', { type: 'button', class: 'btn btn-sm btn-outline-danger' }, 'Descargar XLSX');
    const btnXLS = el('button', { type: 'button', class: 'btn btn-sm btn-outline-secondary' }, 'Descargar XLS');

    const leftControls = el('div', { class: 'd-flex flex-wrap gap-2 align-items-center' },
      ...(extraLeftControls || []),
      el('div', { class: 'd-flex align-items-center gap-2' },
        el('label', { class: 'text-muted small me-1' }, 'Mostrar:'), pageSel
      )
    );
    const rightControls = el('div', { class: 'd-flex flex-wrap gap-2 align-items-center' },
      search, btnXLSX, btnXLS
    );

    const table = el('table', { class: 'table align-middle' },
      el('thead', {}, el('tr', {}, ...columns.map(h => el('th', {}, h)))),
      el('tbody', {})
    );

    const pagerInfo = el('div', { class: 'text-muted small' });
    const btnPrev = el('button', { type: 'button', class: 'btn btn-sm btn-outline-secondary' }, 'Anterior');
    const btnNext = el('button', { type: 'button', class: 'btn btn-sm btn-outline-secondary' }, 'Siguiente');
    const pagerBar = el('div', { class: 'd-flex justify-content-between align-items-center mt-2' },
      pagerInfo, el('div', { class: 'd-flex gap-2' }, btnPrev, btnNext)
    );

    const card = buildCard(title,
      el('div', {},
        el('div', { class: 'd-flex justify-content-between gap-2 mb-2' }, leftControls, rightControls),
        el('div', { class: 'table-responsive' }, table),
        pagerBar
      )
    );

    let pageSize = 50;
    let pageIndex = 0;
    let tokens = [];
    let extraPredicate = null;

    const getFiltered = () => {
      return (data || []).filter(row => {
        const text = normalize(JSON.stringify(row));
        const okTokens = tokens.every(t => text.includes(t));
        const okPred = extraPredicate ? extraPredicate(row) : true;
        return okTokens && okPred;
      });
    };

    const render = () => {
      const list = getFiltered();
      const total = list.length;
      const ps = (pageSize === Infinity) ? total : pageSize;
      const pages = ps ? Math.ceil(total / ps) : 1;
      if (pageIndex >= pages) pageIndex = 0;

      table.tBodies[0].innerHTML = '';
      let start = pageIndex * (ps || 1);
      let end = (pageSize === Infinity) ? total : Math.min(start + ps, total);
      const slice = (pageSize === Infinity) ? list : list.slice(start, end);
      slice.forEach(item => {
        table.tBodies[0].appendChild(renderRow(item, false));
      });

      const from = total ? (start + 1) : 0;
      pagerInfo.textContent = `Mostrando ${from}-${end} de ${total}`;
      btnPrev.disabled = (pageIndex <= 0);
      btnNext.disabled = (pageIndex >= pages - 1 || pages <= 1);
    };

    search.addEventListener('input', () => { tokens = normalize(search.value).split(/\s+/).filter(Boolean); pageIndex = 0; render(); });
    pageSel.addEventListener('change', () => {
      const v = pageSel.value;
      pageSize = (v === 'all') ? Infinity : parseInt(v, 10);
      pageIndex = 0; render();
    });
    btnPrev.addEventListener('click', () => { if (pageIndex > 0) { pageIndex--; render(); } });
    btnNext.addEventListener('click', () => { pageIndex++; render(); });

    // Export: usa todos los datos filtrados (no solo la página actual)
    btnXLSX.addEventListener('click', async () => {
      const rows = tableDataForExport(table.tHead, getFiltered(), renderRow);
      await downloadXLSX(rows, (title || 'reporte') + '.xlsx');
    });
    btnXLS.addEventListener('click', () => {
      const rows = tableDataForExport(table.tHead, getFiltered(), renderRow);
      downloadXLS(rows, (title || 'reporte') + '.xls');
    });

    return {
      card,
      table,
      render,
      setPredicate: (fn) => { extraPredicate = fn; },
      set pageSize(v) { pageSize = v === Infinity ? Infinity : parseInt(v || 50, 10); render(); }
    };
  };

  // ===== Secciones =====
  const buildKpiCards = (root, kpi) => {
    root.appendChild(el('div', { class: 'row g-3' },
      cardKpi('Ingresos (3M)', currency.format(kpi.ingresos_3m || 0), 'Suma de Neto'),
      cardKpi('Unidades (3M)', num(kpi.unidades_3m || 0), ''),
      cardKpi('Órdenes (3M)', num(kpi.ordenes_3m || 0), ''),
      cardKpi('Ticket promedio', currency.format(kpi.ticket_promedio_3m || 0), '')
    ));
  };

  const buildTrendGeneral = (root, rows) => {
    const canvas = el('canvas', { id: 'chartGeneral', height: '110' });
    root.appendChild(buildCard('Tendencia mensual (General)', canvas));
    const labels = rows.map(r => r.mes);
    const ingresos = rows.map(r => +r.ingresos || 0);
    const unidades = rows.map(r => +r.unidades || 0);
    STATE.charts.general = new Chart(canvas.getContext('2d'), {
      type: 'line',
      data: { labels, datasets: [
        { label: 'Ingresos', data: ingresos, yAxisID: 'y1', tension: .3 },
        { label: 'Unidades', data: unidades, yAxisID: 'y2', tension: .3 }
      ]},
      options: { responsive: true, scales: {
        y1: { type: 'linear', position: 'left', beginAtZero: true },
        y2: { type: 'linear', position: 'right', beginAtZero: true }
      }}
    });
  };

  const buildVentasAlmacenCards = (root, rows) => {
    if (!rows || !rows.length) return;
    const grid = el('div', { class: 'row g-3' },
      ...rows.map(r => el('div', { class: 'col-12 col-md-4' },
        el('div', { class: 'card h-100' },
          el('div', { class: 'card-body' },
            el('div', { class: 'd-flex justify-content-between' },
              el('strong', {}, 'Almacén ' + (r.almNombre || '')),
              el('span', { class: 'badge bg-secondary' }, 'ID ' + (r.almId ?? ''))
            ),
            el('div', { class: 'mt-2' },
              el('div', { class: 'text-muted small' }, 'Ingresos 3M'),
              el('div', { class: 'fw-bold fs-5' }, currency.format(r.ingresos || 0))
            ),
            el('div', {},
              el('div', { class: 'text-muted small' }, 'Unidades 3M'),
              el('div', { class: 'fw-bold fs-5' }, num(r.unidades || 0))
            )
          )
        )
      ))
    );
    root.appendChild(buildCard('Ventas por almacén (Resumen 3M)', grid));
  };

  const buildTendenciaAlmacenChart = (root, rows) => {
    const canvas = el('canvas', { id: 'chartAlmacenes', height: '120' });
    root.appendChild(buildCard('Tendencia por almacén', canvas));
    const meses = [...new Set(rows.map(r => r.mes))];
    const almIds = [...new Set(rows.map(r => r.almId))];
    const datasets = almIds.map(id => {
      const serie = meses.map(m => {
        const f = rows.find(x => x.almId == id && x.mes == m);
        return f ? +f.ingresos || 0 : 0;
      });
      const nombre = (rows.find(x => x.almId == id) || {}).almNombre || ('Alm ' + id);
      return { label: nombre, data: serie, type: 'bar', stack: 'ingresos' };
    });
    STATE.charts.almacenes = new Chart(canvas.getContext('2d'), {
      data: { labels: meses, datasets },
      options: { responsive: true, scales: { x: { stacked: true }, y: { stacked: true, beginAtZero: true } } }
    });
  };

  // Top 10 por SKU
  const buildTop10Table = (root, rows) => {
    const ui = makeTableUI({
      title: 'Top 10 productos más vendidos (por SKU)',
      columns: ['Producto', 'SKU', 'Categoría', 'Unidades', 'Ingresos', 'Precio Actual'],
      data: rows,
      renderRow: (r, forExport) => el('tr', {},
        el('td', {}, forExport ? (r.proTitulo || '') :
          el('div', { class: 'd-flex align-items-center gap-3' },
            el('img', { src: r.proImagen || '', class: 'rounded-circle', width: '50', height: '50', alt: '' }),
            el('div', {}, el('div', { class: 'fw-semibold' }, r.proTitulo || ''), el('small', { class: 'text-muted' }, 'Stock: ' + num(r.existencia_total || 0)))
          )
        ),
        el('td', {}, r.proSku || ''),
        el('td', {}, r.categoria || '-'),
        el('td', {}, num(r.unidades_vendidas || 0)),
        el('td', {}, currency.format(r.ingresos || 0)),
        el('td', {}, currency.format(r.precio_actual || 0))
      )
    });
    root.appendChild(ui.card);
    ui.pageSize = 50; ui.render();
  };

  // Sin ventas (3M)
  const buildUnsoldTable = (root, rows) => {
    const ui = makeTableUI({
      title: 'Sin ventas en 3 meses (máx. 10) + recomendaciones',
      columns: ['Producto', 'SKU', 'Categoría', 'Stock', 'Precio Actual', 'Recomendación'],
      data: rows.slice(0, 10),
      renderRow: (u, forExport) => {
        const stock = +u.existencia_total || 0;
        const cat = (u.categoria || '').toLowerCase();
        const recos = [];
        if (stock > 0) {
          recos.push('Descuento -15% (7–14 días)');
          recos.push(cat.includes('extintor') ? 'Bundle con recarga/soporte' : 'Regalo en compra mayor');
          recos.push('Badge OFERTA + anuncio');
        } else {
          recos.push('Revisar SEO o dar de baja');
        }
        return el('tr', {},
          el('td', {}, forExport ? (u.proTitulo || '') :
            el('div', { class: 'd-flex align-items-center gap-3' },
              el('img', { src: u.proImagen || '', class: 'rounded-circle', width: '50', height: '50', alt: '' }),
              el('div', {}, el('div', { class: 'fw-semibold' }, u.proTitulo || ''), el('small', { class: 'text-muted' }, 'Últ. actualización: ' + (u.ultima_actualizacion || '')))
            )
          ),
          el('td', {}, u.proSku || ''),
          el('td', {}, u.categoria || '-'),
          el('td', {}, num(stock)),
          el('td', {}, currency.format(u.precio_actual || 0)),
          el('td', {}, recos.join(' | '))
        );
      }
    });
    root.appendChild(ui.card);
    ui.pageSize = 50; ui.render();
  };

  // Ventas por SKU y almacén (3M)
  const buildVentasSkuAlmacen = (root, rows) => {
    const almacenes = [...new Map(rows.map(r => [r.almId, r.almNombre])).entries()]
      .map(([id, nombre]) => ({ id, nombre }));
    const sel = el('select', { class: 'form-select w-auto' },
      el('option', { value: '' }, 'Todos los almacenes'),
      ...almacenes.map(a => el('option', { value: a.id }, `ID ${a.id} – ${a.nombre}`))
    );

    const ui = makeTableUI({
      title: 'Ventas por SKU y almacén (3M)',
      columns: ['Almacén', 'SKU', 'Producto', 'Unidades', 'Ingresos'],
      data: rows,
      renderRow: (r) => el('tr', {},
        el('td', {}, `${r.almNombre} (ID ${r.almId})`),
        el('td', {}, r.proSku || ''),
        el('td', {}, r.proTitulo || ''),
        el('td', {}, num(r.unidades || 0)),
        el('td', {}, currency.format(r.ingresos || 0))
      ),
      extraLeftControls: [sel]
    });

    ui.setPredicate((row) => !sel.value || String(row.almId) === String(sel.value));
    sel.addEventListener('change', () => ui.render());

    root.appendChild(ui.card);
    ui.pageSize = 50; ui.render();
  };

  // Salud inventario (Global y Por almacén)
  const buildSaludInventario = (root, globalRows, perAlmRows) => {
    // GLOBAL
    const uiG = makeTableUI({
      title: 'Salud de inventario — Global por SKU',
      columns: ['SKU', 'Producto', 'Stock', 'Mín', 'Máx', 'Estatus'],
      data: globalRows,
      renderRow: (r) => {
        const st = +r.stock_total || 0, mi = +r.stock_min_total || 0, ma = +r.stock_max_total || 0;
        const tag = st < mi ? 'Bajo mínimos' : (st > ma ? 'Sobre máximos' : 'OK');
        return el('tr', {},
          el('td', {}, r.proSku || ''),
          el('td', {}, r.proTitulo || ''),
          el('td', {}, num(st)),
          el('td', {}, num(mi)),
          el('td', {}, num(ma)),
          el('td', {}, tag)
        );
      }
    });
    root.appendChild(uiG.card);
    uiG.pageSize = 50; uiG.render();

    // POR ALMACÉN
    const almacenes = [...new Map((perAlmRows || []).map(r => [r.almId, r.almNombre])).entries()]
      .map(([id, nombre]) => ({ id, nombre }));
    const sel = el('select', { class: 'form-select w-auto' },
      el('option', { value: '' }, 'Todos los almacenes'),
      ...almacenes.map(a => el('option', { value: a.id }, `ID ${a.id} – ${a.nombre}`))
    );

    const uiA = makeTableUI({
      title: 'Salud de inventario — Por almacén',
      columns: ['Almacén', 'SKU', 'Producto', 'Stock', 'Mín', 'Máx', 'Estatus'],
      data: perAlmRows,
      renderRow: (r) => {
        const st = +r.stock || 0, mi = +r.stock_min || 0, ma = +r.stock_max || 0;
        const tag = st < mi ? 'Bajo mínimos' : (st > ma ? 'Sobre máximos' : 'OK');
        return el('tr', {},
          el('td', {}, `${r.almNombre} (ID ${r.almId})`),
          el('td', {}, r.proSku || ''),
          el('td', {}, r.proTitulo || ''),
          el('td', {}, num(st)),
          el('td', {}, num(mi)),
          el('td', {}, num(ma)),
          el('td', {}, tag)
        );
      },
      extraLeftControls: [sel]
    });
    uiA.setPredicate((row) => !sel.value || String(row.almId) === String(sel.value));
    sel.addEventListener('change', () => uiA.render());
    root.appendChild(uiA.card);
    uiA.pageSize = 50; uiA.render();
  };

  // Top vendidos con stock/min/max (con id para PDF)
  const buildTopVendidosStockTable = (root, rows) => {
    const ui = makeTableUI({
      title: 'Productos más vendidos (3M) con stock / mínimos / máximos',
      columns: ['Producto', 'SKU', 'Categoría', 'Unidades (3M)', 'Stock', 'Mín', 'Máx'],
      data: rows,
      renderRow: (r) => el('tr', {},
        el('td', {}, r.proTitulo || ''),
        el('td', {}, r.proSku || ''),
        el('td', {}, r.categoria || '-'),
        el('td', {}, num(r.unidades_vendidas || 0)),
        el('td', {}, num(r.stock_total || 0)),
        el('td', {}, num(r.stock_min_total || 0)),
        el('td', {}, num(r.stock_max_total || 0))
      )
    });
    root.appendChild(ui.card);
    // Añadir id al <table> para PDF
    ui.table.id = 'tblTopVendidos';
    ui.pageSize = 50; ui.render();
  };

  // Sin ventas + kits sugeridos (con id para PDF)
  const buildKitsFromUnsoldTable = (root, unsold, topVendidos) => {
    const topByCat = {};
    (topVendidos || []).forEach(t => {
      const k = (t.categoria || '').toLowerCase();
      if (!topByCat[k]) topByCat[k] = [];
      topByCat[k].push(t);
    });
    const complementos = [
      { ifCat: /extintor/i,  kit: 'Extintor + Soporte + Señalización + Recarga' },
      { ifCat: /soporte/i,   kit: 'Soporte + Extintor 6kg + Tornillería' },
      { ifCat: /señal/i,     kit: 'Señal + Extintor + Kit emergencia' },
      { ifCat: /botiqu/i,    kit: 'Botiquín + Guantes + Alcohol en gel' },
      { ifCat: /seguridad/i, kit: 'Casco + Lentes + Chaleco' }
    ];
    const suggestKit = (row) => {
      const cat = (row.categoria || '').toLowerCase();
      const reg = complementos.find(x => x.ifCat.test(cat));
      if (reg) return reg.kit;
      const peer = (topByCat[cat] || [])[0];
      if (peer) return `${row.proTitulo} + ${peer.proTitulo}`;
      return 'Bundle como regalo o promo -15% (7–14 días)';
    };

    const ui = makeTableUI({
      title: 'Productos sin ventas (3M) + kits sugeridos',
      columns: ['Producto', 'SKU', 'Categoría', 'Stock', 'Mín', 'Máx', 'Kit sugerido'],
      data: unsold,
      renderRow: (r) => el('tr', {},
        el('td', {}, r.proTitulo || ''),
        el('td', {}, r.proSku || ''),
        el('td', {}, r.categoria || '-'),
        el('td', {}, num(r.existencia_total || 0)),
        el('td', {}, num(r.stock_min_total || 0)),
        el('td', {}, num(r.stock_max_total || 0)),
        el('td', {}, suggestKit(r))
      )
    });
    root.appendChild(ui.card);
    // Añadir id al <table> para PDF
    ui.table.id = 'tblKits';
    ui.pageSize = 50; ui.render();
  };

  const buildPromos = (root, rows) => {
    if (!rows || !rows.length) return;
    const grid = el('div', { class: 'row g-3' },
      ...rows.map(pr => el('div', { class: 'col-12 col-md-6 col-lg-4' },
        el('div', { class: 'card h-100' },
          el('img', { src: pr.proImagen || '', class: 'card-img-top', style: 'height:170px;object-fit:cover' }),
          el('div', { class: 'card-body' },
            el('h6', { class: 'card-title' }, pr.proTitulo || ''),
            el('div', {},
              el('span', { class: 'badge bg-danger text-dark text-decoration-line-through me-2' }, currency.format(pr.proPrecioventa || 0)),
              el('span', { class: 'badge bg-success' }, currency.format(pr.preciopromo || 0))
            ),
            el('small', { class: 'text-muted d-block mt-1' }, `Vigencia: ${pr.FechaInicial || ''} a ${pr.FechaFinal || ''}`)
          )
        )
      ))
    );
    root.appendChild(buildCard('Promociones activas', grid));
  };

  const wirePdfDownload = () => {
    const btn = $('#btnDescargarPdf'); if (!btn) return;
    btn.addEventListener('click', () => {
      try {
        $('#chart_general').value = STATE.charts.general ? STATE.charts.general.toBase64Image() : '';
        $('#chart_almacenes').value = STATE.charts.almacenes ? STATE.charts.almacenes.toBase64Image() : '';
        const topTbl = $('#tblTopVendidos');
        const kitsTbl = $('#tblKits');
        $('#tabla_top').value = topTbl ? topTbl.outerHTML : '';
        $('#tabla_kits').value = kitsTbl ? kitsTbl.outerHTML : '';
        $('#pdfForm').submit();
      } catch (e) {
        alert('No se pudo preparar el PDF: ' + e.message);
      }
    });
  };

  // ===== Main =====
  document.addEventListener('DOMContentLoaded', async () => {
    const root = $(TARGET);
    if (!root) { console.error('No se encontró #kpi-card-body'); return; }

    buildToolbar(root);
    const spinner = buildSpinner(root);

    try {
      await loadScriptIfMissing('Chart', 'https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js');

      const [
        kpiRes, trendRes, ventasAlmacenRes, tendenciaAlmacenRes,
        top10Res, unsoldRes, promosRes, ventasSkuAlmacenRes,
        saludGlobalRes, saludAlmacenRes, topVendidosStockRes, unsoldStockRes
      ] = await Promise.all([
        apiPost(ENDPOINTS.kpiGlobal),
        apiPost(ENDPOINTS.trendMensual),
        apiPost(ENDPOINTS.ventasAlmacen),
        apiPost(ENDPOINTS.tendenciaAlmacen),
        apiPost(ENDPOINTS.top10Sku),
        apiPost(ENDPOINTS.sinVentas),
        apiPost(ENDPOINTS.promos),
        apiPost(ENDPOINTS.ventasSkuAlmacen),
        apiPost(ENDPOINTS.saludSkuGlobal),
        apiPost(ENDPOINTS.saludSkuAlmacen),
        apiPost(ENDPOINTS.topVendidosStock),
        apiPost(ENDPOINTS.unsoldStock)
      ]);

      STATE.data = {
        kpi:               (kpiRes && kpiRes[0]) || { ingresos_3m: 0, unidades_3m: 0, ordenes_3m: 0, ticket_promedio_3m: 0 },
        trend:             trendRes || [],
        ventasAlmacen:     ventasAlmacenRes || [],
        tendenciaAlmacen:  tendenciaAlmacenRes || [],
        top10:             top10Res || [],
        unsold:            unsoldRes || [],
        promos:            promosRes || [],
        ventasSkuAlmacen:  ventasSkuAlmacenRes || [],
        saludGlobal:       saludGlobalRes || [],
        saludAlmacen:      saludAlmacenRes || [],
        topVendidosStock:  topVendidosStockRes || [],
        unsoldStock:       unsoldStockRes || []
      };

      spinner.remove();

      buildKpiCards(root, STATE.data.kpi);
      buildTrendGeneral(root, STATE.data.trend);
      buildVentasAlmacenCards(root, STATE.data.ventasAlmacen);
      buildTendenciaAlmacenChart(root, STATE.data.tendenciaAlmacen);
      buildTop10Table(root, STATE.data.top10);
      buildUnsoldTable(root, STATE.data.unsold);
      buildVentasSkuAlmacen(root, STATE.data.ventasSkuAlmacen);
      buildSaludInventario(root, STATE.data.saludGlobal, STATE.data.saludAlmacen);
      buildTopVendidosStockTable(root, STATE.data.topVendidosStock);
      buildKitsFromUnsoldTable(root, STATE.data.unsoldStock, STATE.data.topVendidosStock);
      buildPromos(root, STATE.data.promos);

      wirePdfDownload();

    } catch (e) {
      console.error(e);
      spinner.replaceWith(
        el('div', { class: 'alert alert-danger' },
          'No se pudieron cargar los reportes. ',
          el('small', { class: 'd-block text-muted mt-1' }, e.message)
        )
      );
    }
  });
})();
