/* productosaeliminar.js - Página "Productos a eliminar" */

(() => {
  const API_BASE = 'https://grupoxtinfire.com/admin/ajax/';
  const HOOK = 'codi_hook';
  const ACTION = 'load';
  const ROOT = '#productos-eliminar-card-body';

  const ENDPOINTS = {
    unsoldSku6m:         'unsold_sku_6m',
    unsoldSkuAlmacen6m:  'unsold_sku_almacen_6m',
    topCat3m:            'top_sellers_by_category_3m'
  };

  const $ = (sel, ctx=document) => ctx.querySelector(sel);

  // ---------- helpers ----------
  const el = (tag, attrs, ...children) => {
    const node = document.createElement(tag);
    if (attrs && typeof attrs === 'object') {
      for (const [k,v] of Object.entries(attrs)) {
        if (v===undefined || v===null || v===false) continue;
        if (k==='class') node.className = v;
        else if (k==='dataset') Object.assign(node.dataset, v);
        else if (k.startsWith('on') && typeof v==='function') node.addEventListener(k.slice(2), v);
        else node.setAttribute(k, v);
      }
    }
    const flat = children.flat ? children.flat() : [].concat(...children);
    flat.forEach(c=>{
      if (c===undefined || c===null || c===false) return;
      if (typeof c==='string' || typeof c==='number') node.appendChild(document.createTextNode(String(c)));
      else if (c && typeof c==='object' && 'nodeType' in c) node.appendChild(c);
      else node.appendChild(document.createTextNode(String(c)));
    });
    return node;
  };

  const currency = new Intl.NumberFormat('es-MX',{style:'currency',currency:'MXN'});
  const num = (n,d=0)=> (Number(n)||0).toLocaleString('es-MX',{minimumFractionDigits:d,maximumFractionDigits:d});
  const normalize = s => (s||'').toString().toLowerCase().normalize('NFD').replace(/\p{Diacritic}/gu,'');

  const buildSpinner = (root) => root.appendChild(
    el('div',{class:'d-flex justify-content-center my-3'},
      el('div',{class:'spinner-border text-danger me-2', role:'status'}),'Cargando...')
  );

  const loadScriptIfMissing = (globalName, src) => new Promise((resolve,reject)=>{
    if (window[globalName]) return resolve();
    const s = el('script',{src,async:true});
    s.onload = resolve;
    s.onerror = () => reject(new Error('No se pudo cargar: '+src));
    document.head.appendChild(s);
  });

  const ensureChart = () => loadScriptIfMissing('Chart','https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js');
  const ensureXLSX = () => new Promise((resolve,reject)=>{
    if (window.XLSX) return resolve();
    const urls = [
      'https://cdn.sheetjs.com/xlsx-0.20.3/package/dist/xlsx.full.min.js',
      'https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js',
      'https://unpkg.com/xlsx@0.18.5/dist/xlsx.full.min.js'
    ];
    (function tryNext(i){
      if (i>=urls.length) return reject(new Error('No se pudo cargar XLSX'));
      const s = el('script',{src:urls[i],async:true});
      s.onload = resolve;
      s.onerror= ()=> tryNext(i+1);
      document.head.appendChild(s);
    })(0);
  });

  const apiPost = async (fnName, params={})=>{
    const body = new URLSearchParams({hook:HOOK, action:ACTION, ...params});
    const res = await fetch(API_BASE + fnName, {
      method:'POST',
      headers:{'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8'},
      body
    });
    if (!res.ok) throw new Error('HTTP '+res.status+' @'+fnName);
    const json = await res.json();
    if (!json || json.status !== 200 || json.error) throw new Error('API '+fnName+' error');
    return json.data || [];
  };

  // Tabla con buscador, paginación y export; ahora soporta colWidths y modo compacto
const makeTableUI = ({ title, columns, data, renderRow, extraLeftControls=[], colWidths=null, compact=false }) => {
  const search = el('input',{type:'text',class:'form-control',placeholder:'Buscar (multi-palabra)'});
  const pageSel = el('select',{class:'form-select'},
    el('option',{value:'50'},'50'),
    el('option',{value:'100'},'100'),
    el('option',{value:'all'},'Todos')
  );
  const btnXLSX = el('button',{type:'button',class:'btn btn-sm btn-outline-danger'},'XLSX');
  const btnXLS  = el('button',{type:'button',class:'btn btn-sm btn-outline-danger'},'XLS');

  const left = el('div',{class:'d-flex flex-wrap gap-2 align-items-center'}, ...extraLeftControls,
    el('div',{class:'d-flex align-items-center gap-2'}, el('label',{class:'small text-muted'},'Mostrar:'), pageSel)
  );
  const right = el('div',{class:'d-flex flex-wrap gap-2 align-items-center'}, search, btnXLSX, btnXLS);

  // colgroup opcional para fijar anchos
  const colgroup = colWidths ? el('colgroup',{},
    ...colWidths.map((w,i)=> el('col',{style: w ? `width:${w}` : '', class: (columns[i]==='Recomendación' ? 'reco-col' : '')}))
  ) : null;

  const table = el('table',{
    class: `table align-middle ${compact ? 'table-sm table-compact table-fixed' : ''}`, 'data-table':''
  },
    colgroup ? colgroup : [],
    el('thead',{}, el('tr',{}, ...columns.map(h=>el('th',{},h)))),
    el('tbody',{})
  );

  const info = el('div',{class:'text-muted small'});
  const prev = el('button',{type:'button',class:'btn btn-sm btn-outline-danger'},'Anterior');
  const next = el('button',{type:'button',class:'btn btn-sm btn-outline-danger'},'Siguiente');

  const card = el('div',{class:'card mt-3'},
    el('div',{class:'card-body'},
      el('div',{class:'d-flex justify-content-between align-items-center mb-2'},
        el('h6',{class:'m-0'}, title),
        el('div',{},)
      ),
      el('div',{class:'d-flex justify-content-between gap-2 mb-2'}, left, right),
      el('div',{class:'table-responsive'}, table),
      el('div',{class:'d-flex justify-content-between align-items-center'}, info, el('div',{class:'d-flex gap-2'}, prev, next))
    )
  );

  let pageSize = 50, pageIndex = 0, tokens=[];
  const normalize = s => (s||'').toString().toLowerCase().normalize('NFD').replace(/\p{Diacritic}/gu,'');
  const normalizeRow = r => normalize(JSON.stringify(r));
  const getFiltered = () => (data||[]).filter(r => tokens.every(t=> normalizeRow(r).includes(t)));

  const render = ()=>{
    const list = getFiltered();
    const total = list.length;
    const ps = pageSize===Infinity ? total : pageSize;
    const pages = ps ? Math.ceil(total/ps) : 1;
    if (pageIndex >= pages) pageIndex = 0;

    table.tBodies[0].innerHTML='';
    const start = pageIndex * (ps||1);
    const end   = pageSize===Infinity ? total : Math.min(start+ps, total);
    const slice = pageSize===Infinity ? list : list.slice(start,end);
    slice.forEach(row=> table.tBodies[0].appendChild(renderRow(row)));

    const from = total ? (start+1) : 0;
    info.textContent = `Mostrando ${from}-${end} de ${total}`;
    prev.disabled = (pageIndex<=0); next.disabled=(pageIndex>=pages-1 || pages<=1);
  };

  search.addEventListener('input', ()=>{ tokens=normalize(search.value).split(/\s+/).filter(Boolean); pageIndex=0; render(); });
  pageSel.addEventListener('change', ()=>{ const v=pageSel.value; pageSize=(v==='all')?Infinity:parseInt(v,10); pageIndex=0; render(); });
  prev.addEventListener('click', ()=>{ if(pageIndex>0){ pageIndex--; render(); }});
  next.addEventListener('click', ()=>{ pageIndex++; render(); });

  // export
  const tableDataForExport = ()=>{
    const headers = [...table.tHead.querySelectorAll('th')].map(th=>th.textContent.trim());
    const rows = getFiltered().map(r=>{
      const tr = renderRow(r);
      return [...tr.children].map(td=>td.textContent.trim());
    });
    return [headers, ...rows];
  };
  const ensureXLSX = async ()=> {
    if (window.XLSX) return;
    const s = document.createElement('script');
    s.src = 'https://cdn.sheetjs.com/xlsx-0.20.3/package/dist/xlsx.full.min.js';
    await new Promise((ok,err)=>{ s.onload=ok; s.onerror=()=>err(new Error('XLSX no cargó')); document.head.appendChild(s); });
  };
  btnXLSX.addEventListener('click', async ()=>{
    try{ await ensureXLSX(); const rows = tableDataForExport();
      const ws = XLSX.utils.aoa_to_sheet(rows); const wb = XLSX.utils.book_new();
      XLSX.utils.book_append_sheet(wb, ws, 'Datos'); XLSX.writeFile(wb, (title||'reporte')+'.xlsx');
    }catch(e){ alert('No se pudo exportar XLSX: '+e.message); }
  });
  btnXLS.addEventListener('click', ()=>{
    const rows = tableDataForExport();
    const html = `<html><head><meta charset="utf-8"></head><body><table border="1">${rows.map(r=>'<tr>'+r.map(c=>`<td>${String(c).replace(/</g,'&lt;')}</td>`).join('')+'</tr>').join('')}</table></body></html>`;
    const blob = new Blob([html], {type:'application/vnd.ms-excel'});
    const a=document.createElement('a'); a.download=(title||'reporte')+'.xls'; a.href=URL.createObjectURL(blob);
    document.body.appendChild(a); a.click(); URL.revokeObjectURL(a.href); a.remove();
  });

  return { card, table, render };
};


  const ensureModal = (id, title='Detalle')=>{
    let modal = document.getElementById(id);
    if (!modal) {
      modal = el('div',{class:'modal fade', id, tabindex:'-1'},
        el('div',{class:'modal-dialog modal-xl modal-dialog-scrollable'},
          el('div',{class:'modal-content'},
            el('div',{class:'modal-header'},
              el('h5',{class:'modal-title'}, title),
              el('button',{type:'button',class:'btn-close','data-bs-dismiss':'modal','aria-label':'Close'})
            ),
            el('div',{class:'modal-body', id:id+'-body'}, 'Cargando...'),
            el('div',{class:'modal-footer'},
              el('button',{type:'button',class:'btn btn-danger','data-bs-dismiss':'modal'},'Cerrar')
            )
          )
        )
      );
      document.body.appendChild(modal);
    }
    return modal;
  };

  // === CSS compacto para tablas de esta página ===
(() => {
  if (!document.getElementById('pe-compact-css')) {
    const style = document.createElement('style');
    style.id = 'pe-compact-css';
    style.textContent = `
      /* compacta filas y mejora legibilidad en dark/blank */
      table.table-compact td, table.table-compact th { padding: .4rem .55rem; vertical-align: middle; }
      /* evita que la columna de kits salte de línea; aparece scroll si hay muchos */
      .kits-nowrap { display:flex; gap:.35rem; flex-wrap:nowrap; overflow:auto; padding-bottom:2px; }
      /* badges rojos para dark theme */
      .badge-danger-compact { font-weight:500; }
      /* controla anchos por colgroup */
      .table-fixed { table-layout: fixed; }
      .wrap-normal { white-space: normal; word-break: break-word; }
      /* Rec: que ocupe más ancho sin romper el layout */
      col.reco-col { width: 28% !important; }
    `;
    document.head.appendChild(style);
  }
})();

  // ---------- recomendaciones ----------
  const buildRecoText = (row) => {
    const ex = Number(row.existencia_total||0);
    const min = Number(row.min_ref||0);
    const last = row.ultima_actualizacion || '';
    const cat = row.categoria || '';
    const base = [];
    if (ex > 0 && ex >= Math.max(min,1)) base.push('Sobrestock sin rotación (6M).');
    if (last) base.push(`Última actualización: ${last}.`);
    base.push('Sugerencias: 1) bundle con top ventas de la categoría, 2) 2x1 / 3x2 limitado, 3) precio escalonado por volumen, 4) regalo en compras +$X.');
    if (cat) base.push(`Probar combo dentro de "${cat}".`);
    return base.join(' ');
  };

  // ---------- secciones ----------
  const buildKPIs = (root, rows)=>{
    const skus = (rows||[]).length;
    const ex   = (rows||[]).reduce((a,b)=>a+Number(b.existencia_total||0),0);
    const val  = (rows||[]).reduce((a,b)=>a+Number(b.valor_inventario||0),0);
    const row = el('div',{class:'row g-3'},
      el('div',{class:'col-12 col-md-4'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'SKUs candidatos'),
          el('div',{class:'fw-bold fs-5'}, num(skus))
        )
      ),
      el('div',{class:'col-12 col-md-4'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Existencias totales'),
          el('div',{class:'fw-bold fs-5'}, num(ex))
        )
      ),
      el('div',{class:'col-12 col-md-4'},
        el('div',{class:'p-3 bg-light rounded-3 h-100'},
          el('div',{class:'small text-muted'},'Valor de inventario'),
          el('div',{class:'fw-bold fs-5'}, currency.format(val))
        )
      )
    );
    root.appendChild(row);
  };

  const buildCategoryChart = (root, rows)=>{
    const byCat = {};
    (rows||[]).forEach(r=>{
      const k = r.categoria || '(Sin categoría)';
      byCat[k] = (byCat[k]||0) + Number(r.valor_inventario||0);
    });
    const labels = Object.keys(byCat);
    const values = labels.map(k=> byCat[k]);
    const canvas = el('canvas',{id:'chartEliminarCat', height:'110'});
    root.appendChild(el('div',{class:'card mt-3'}, el('div',{class:'card-body'},
      el('h6',{},'Valor de inventario por categoría (candidatos)'),
      canvas
    )));
    new Chart(canvas.getContext('2d'),{
      type:'bar',
      data:{ labels, datasets:[{label:'Valor', data:values}]},
      options:{responsive:true, scales:{y:{beginAtZero:true}}}
    });
  };

 const buildMainTable = (root, rows, detalle, topCat)=>{
  const topByCat = {};
  (topCat||[]).forEach(r=>{
    const cat = r.categoria || '(Sin categoría)';
    if (!topByCat[cat]) topByCat[cat] = [];
    if (topByCat[cat].length < 3) topByCat[cat].push({sku:r.proSku, titulo:r.proTitulo});
  });

  const modal = ensureModal('modal-detalle-alm','Detalle por almacén (SKU)');
  const modalBody = document.getElementById('modal-detalle-alm-body');

  const verDetalle = (sku)=>{
    const rowsSku = (detalle||[]).filter(d=> String(d.proSku)===String(sku));
    const ui = makeTableUI({
      title:`Detalle por almacén — ${sku}`,
      columns:['Almacén','proId','Título','Existencia','Precio','Valor','Última actualización'],
      data: rowsSku,
      compact:true,
      renderRow: r => el('tr',{},
        el('td',{}, `${r.almId} - ${r.almNombre}`),
        el('td',{}, r.proId),
        el('td',{}, r.proTitulo || ''),
        el('td',{}, (Number(r.proExistencia||0)).toLocaleString('es-MX')),
        el('td',{}, (new Intl.NumberFormat('es-MX',{style:'currency',currency:'MXN'})).format(r.proPrecioventa||0)),
        el('td',{}, (new Intl.NumberFormat('es-MX',{style:'currency',currency:'MXN'})).format(r.valor_inventario||0)),
        el('td',{}, r.FechaActualizacion || '')
      )
    });
    modalBody.innerHTML = ''; modalBody.appendChild(ui.card); ui.render();
    new bootstrap.Modal(modal).show();
  };

  const renderKits = (cat)=>{
    const kits = (topByCat[cat]||[]);
    if (!kits.length) return document.createTextNode('—');
    const wrap = el('div',{class:'kits-nowrap'});
    kits.forEach(k=>{
      wrap.appendChild(el('span',{class:'badge text-bg-danger badge-danger-compact'}, k.titulo || k.sku));
    });
    return wrap;
  };

  // Más espacio para “Recomendación”
  const colWidths = ['10%','18%','12%','8%','10%','28%','9%','5%'];

  const ui = makeTableUI({
    title:'Candidatos a eliminar (sin ventas 6M)',
    columns:['SKU','Título','Categoría','Existencia','Valor','Recomendación','Kits sugeridos','Acciones'],
    colWidths,
    compact:true,
    data: rows || [],
    renderRow: r => {
      const currency = new Intl.NumberFormat('es-MX',{style:'currency',currency:'MXN'});
      const titulo = r.titulo || r.proSku;

      const acciones = el('div',{class:'d-flex gap-2'},
        el('button',{type:'button',class:'btn btn-sm btn-danger', onclick:()=>verDetalle(r.proSku)},'Ver almacenes')
      );

      const reco = el('div',{class:'wrap-normal'}, buildRecoText(r));

      return el('tr',{},
        el('td',{}, r.proSku),
        el('td',{}, titulo),
        el('td',{}, r.categoria || ''),
        el('td',{}, (Number(r.existencia_total||0)).toLocaleString('es-MX')),
        el('td',{}, currency.format(r.valor_inventario||0)),
        el('td',{}, reco),
        el('td',{}, renderKits(r.categoria || '(Sin categoría)')),
        el('td',{}, acciones)
      );
    }
  });

  ui.table.setAttribute('data-table','candidatos-eliminar');
  root.appendChild(ui.card);
  ui.render();
};



  const buildPdfToolbar = (root)=>{
    const form = el('form',{id:'pdfEliminarForm', method:'post', action:'https://grupoxtinfire.com/admin/pdfreportes/productos_eliminar_pdf', class:'d-flex align-items-center gap-2 mb-3'},
      el('input',{type:'hidden',name:'chart_categorias',id:'chart_categorias'}),
      el('input',{type:'hidden',name:'tabla_candidatos',id:'tabla_candidatos'}),
      el('input',{type:'hidden',name:'tabla_detalle_stub',id:'tabla_detalle_stub'}), /* opcional */
      el('button',{type:'button',id:'btnPdfEliminar',class:'btn btn-sm btn-dark'},
        el('span',{class:'material-icons-outlined align-middle me-1'},'picture_as_pdf'),'Descargar PDF'
      )
    );
    root.appendChild(form);

    $('#btnPdfEliminar').addEventListener('click', ()=>{
      try{
        const ch = document.getElementById('chartEliminarCat');
        $('#chart_categorias').value = ch ? ch.toDataURL('image/png') : '';
        const tbl = document.querySelector('#productos-eliminar-card-body table[data-table="candidatos-eliminar"]');
        $('#tabla_candidatos').value = tbl ? tbl.outerHTML : '';
        $('#pdfEliminarForm').submit();
      }catch(e){ alert('No se pudo preparar el PDF: '+e.message); }
    });
  };

  // ---------- main ----------
  document.addEventListener('DOMContentLoaded', async ()=>{
    const root = $(ROOT); if (!root) return;
    buildPdfToolbar(root);
    const spin = buildSpinner(root);

    try{
      await ensureChart();

      const [unsold, detalle, topcat] = await Promise.all([
        apiPost(ENDPOINTS.unsoldSku6m),
        apiPost(ENDPOINTS.unsoldSkuAlmacen6m),
        apiPost(ENDPOINTS.topCat3m)
      ]);

      spin.remove();

      buildKPIs(root, unsold||[]);
      buildCategoryChart(root, unsold||[]);
      buildMainTable(root, unsold||[], detalle||[], topcat||[]);

    }catch(e){
      console.error(e);
      spin.replaceWith(el('div',{class:'alert alert-danger'}, 'No se pudieron cargar los reportes. ', el('small',{class:'d-block text-muted mt-1'}, e.message)));
    }
  });
})();
