/* =====================================
   HELPERS GLOBALES
   ===================================== */
function setDisabledWithin(el, disabled) {
  if (!el) return;
  el.querySelectorAll("input, select, textarea, button").forEach(c => c.disabled = disabled);
}
function normalizeBasic(s) {
  return String(s || '')
    .toLowerCase()
    .normalize('NFD').replace(/[\u0300-\u036f]/g, '');
}

/* =====================================
REPORTES por DEPARTAMENTO
===================================== */
const REPORTES = {
  "Almacen": [
    { label: "Catálogo de productos", endpoint: "pdfcatalogo" },
    { label: "Inventarios", endpoint: "pdfinvetarios" },
    { label: "Inventarios tienda", endpoint: "pdfinvetariostienda" },
    { label: "Compras", endpoint: "pdfcompras" },
    { label: "Almacenes", endpoint: "pdfAlmacenes" },
    { label: "Vales", endpoint: "almacen-vales" },
    { label: "Categorías", endpoint: "pdfCategorias" },
    { label: "Traspasos", endpoint: "pdftraspasos" },
    { label: "Valuación de inventario", endpoint: "almacen-valuacion" }
  ],
  "Ventas": [
    { label: "Ventas Generales", endpoint: "pdfVentasGenerales" },
    { label: "Ventas por Producto", endpoint: "pdfVentasPorProducto" },
    { label: "Ventas por categoría", endpoint: "pdfVentasPorCategoria" },
    { label: "Ventas por vendedores", endpoint: "pdfVentasPorVendedor" },
    { label: "Devoluciones", endpoint: "ventas-devoluciones" }
  ],
  "PIPC": [
    { label: "Iniciados", endpoint: "pipcReportes" },
    { label: "Atrasados", endpoint: "pipc-atrasados" },
    { label: "Pendientes por pagar", endpoint: "pipc-pendientes-pago" },
    { label: "Finalizados", endpoint: "pipc-finalizados" }
  ],
  /* ====== Capacitaciones SOLO con 2 opciones ====== */
  "Capacitaciones": [
    { label: "Desglose capacitaciones", endpoint: "pdfDesgloceCapacitaciones" },
    { label: "Asistencias",              endpoint: "pdfAsitenciasCapacitaciones" }
  ],
  /* Reparto seguirá listado aquí para compatibilidad, pero se reemplaza por checkboxes */
  "Reparto": [
    { label: "Pendientes por asignar", endpoint: "reparto-pendientes-asignar" },
    { label: "Asignadas", endpoint: "reparto-asignadas" },
    { label: "Asignadas por repartidor", endpoint: "reparto-asignadas-por-repartidor" },
    { label: "Devueltos Almacén", endpoint: "reparto-devueltos-almacen" },
    { label: "Entregadas", endpoint: "reparto-entregadas" }
  ],
  "Movimientos por usuario": [
    { label: "Todos los usuarios", endpoint: "mov-usuarios-todos" },
    { label: "Por usuario", endpoint: "mov-usuarios-por-usuario" },
    { label: "Por departamento", endpoint: "mov-usuarios-por-departamento" },
    { label: "Por acción", endpoint: "mov-usuarios-por-accion" }
  ],
  "Tienda": [
    { label: "Ventas", endpoint: "tienda-ventas" },
    { label: "Ventas por usuario", endpoint: "tienda-ventas-por-usuario" },
    { label: "Top productos", endpoint: "tienda-top-productos" }
  ]
};

/* Endpoints CAP específicos */
const CAP_ENDPOINTS = {
  DESGLOSE: "pdfDesgloceCapacitaciones",
  ASISTENCIAS: "pdfAsitenciasCapacitaciones"
};

/* Endpoint único para Reparto (el backend leerá los checkboxes) */
const REPARTO_ENDPOINT = "repartoReportes";

/* =====================================
OPCIONES AVANZADAS (definición + reglas + defaults)
===================================== */
const ADVANCED_OPTIONS_MASTER = [
  { name: "existencias",         label: "Incluir existencias" },
  { name: "costo_compra",        label: "Mostrar costo de compra" },
  { name: "max_min",             label: "Máximos y mínimos" },
  { name: "mostrar_descripcion", label: "Mostrar descripción" },
  { name: "mostrar_sku",         label: "Mostrar SKU" },
  { name: "mostrar_imagen",      label: "Mostrar imagen" },
  { name: "mostrar_precio",      label: "Mostrar precio de venta" }
];

/* Alinear claves con endpoints reales usados en REPORTES */
const ADVANCED_RULES = {
  // ===== ALMACÉN =====
  "pdfcatalogo": ["existencias","mostrar_descripcion","mostrar_sku","mostrar_imagen","mostrar_precio","max_min"],
  "pdfinvetarios": ["existencias","max_min","mostrar_sku"],
  "pdfinvetariostienda": ["existencias","max_min","mostrar_sku"],
  "pdfcompras": [],
  "pdfAlmacenes": ["mostrar_descripcion"],
  "pdfCategorias": ["mostrar_descripcion"],
  "pdftraspasos": ["mostrar_sku"],
  "almacen-valuacion": ["existencias","costo_compra","max_min"],

  // ===== VENTAS =====
  "pdfVentasGenerales": [],
  "pdfVentasPorProducto": ["mostrar_sku","mostrar_imagen"],
  "pdfVentasPorCategoria": ["mostrar_descripcion"],
  "pdfVentasPorVendedor": [],
  "ventas-devoluciones": ["mostrar_descripcion","mostrar_sku"],

  // ===== PIPC =====
  "pipcReportes": [],
  "pipc-atrasados": [],
  "pipc-pendientes-pago": [],
  "pipc-finalizados": [],

  // ===== CAPACITACIONES (no usamos el bloque estándar en CAP) =====
  "pdfDesgloceCapacitaciones": [],
  "pdfAsitenciasCapacitaciones": [],

  // ===== REPARTO (endpoint único; las opciones estándar no aplican)
  "repartoReportes": [],

  // ===== MOVIMIENTOS =====
  "mov-usuarios-todos": [],
  "mov-usuarios-por-usuario": [],
  "mov-usuarios-por-departamento": [],
  "mov-usuarios-por-accion": [],

  // ===== TIENDA =====
  "tienda-ventas": [],
  "tienda-ventas-por-usuario": [],
  "tienda-top-productos": ["mostrar_imagen","mostrar_sku"]
};

const DEFAULTS_BY_ENDPOINT = {
  "pdfcatalogo": ["mostrar_descripcion","mostrar_sku","mostrar_precio"],
  "pdfinvetarios": ["existencias","max_min","mostrar_sku"],
  "pdfinvetariostienda": ["existencias","max_min","mostrar_sku"],
  "pdfAlmacenes": ["mostrar_descripcion"],
  "pdfCategorias": ["mostrar_descripcion"],
  "pdftraspasos": ["mostrar_sku"],
  "almacen-valuacion": ["existencias","costo_compra"],
  "pdfVentasPorProducto": ["mostrar_sku"],
  "pdfVentasPorCategoria": ["mostrar_descripcion"],
  "ventas-devoluciones": ["mostrar_descripcion","mostrar_sku"],
  "tienda-top-productos": ["mostrar_imagen","mostrar_sku"]
};

/* =====================================
CATEGORÍAS (se reemplazan por API)
===================================== */
const CATEGORIAS = [
  { id: 'local-1', nombre: 'Recargas y mantenimiento' },
  { id: 'local-2', nombre: 'Alarmas' },
  { id: 'local-3', nombre: 'PIPC' },
  { id: 'local-4', nombre: 'Refacciones' },
  { id: 'local-5', nombre: 'Accesorios' },
  { id: 'local-6', nombre: 'Cámaras' }
];

/* =====================================
ELEMENTOS DOM
===================================== */
const form = document.getElementById("formReportes");
const selDepto = document.getElementById("selDepto");
const selReporte = document.getElementById("selReporte");
const actionPreview = document.getElementById("actionPreview");

const categoriasCard = document.getElementById("categoriasCard");
const categoriasWrap = document.getElementById("categoriasWrap");
const categoriasNotice = document.getElementById("categoriasNotice");

/* Checkbox “Buscar por fecha agendada” (solo CAP → Desglose) */
const capFechaAgendaWrap = document.getElementById("capFechaAgendaWrap");
const capFechaAgenda = document.getElementById("capFechaAgenda");

const advCard = document.getElementById("advCard");
const optsAvanzadas = document.getElementById("optsAvanzadas");
const avisoOpciones = document.getElementById("avisoOpciones");

const advExtrasVPP = document.getElementById("advExtrasVPP");
const vppSku = document.getElementById("vppSku");
const vppSkuWrap = vppSku ? (vppSku.closest('.col-12.col-md-6') || vppSku.closest('.col-12')) : null;
const vppAlmacen = document.getElementById("vppAlmacen");
const vppUltima = document.getElementById("vppUltima");
const vppUltimaWrap = document.getElementById("vppUltimaWrap");

const advExtrasVENTAS_FILTROS = document.getElementById("advExtrasVENTAS_FILTROS");
const venEstatus = document.getElementById("venEstatus");
const venTransito = document.getElementById("venTransito");

const vppAlmacenHelp = document.getElementById("vppAlmacenHelp");

const advExtrasTR = document.getElementById("advExtrasTR");
const traspAlmacen = document.getElementById("traspAlmacen");
const traspAlmacenHelp = document.getElementById("traspAlmacenHelp");

// --- PIPC extras
const advExtrasPIPC = document.getElementById("advExtrasPIPC");
const selReporteCol = selReporte ? (selReporte.closest(".col-12.col-md-5") || selReporte.closest(".col-12")) : null;
const PIPC_ENDPOINT = "pipcReportes";

/* NUEVO inventarios almacén */
const advExtrasALM_INV = document.getElementById("advExtrasALM_INV");
const invAlmacen = document.getElementById("invAlmacen");
const invAlmacenHelp = document.getElementById("invAlmacenHelp");

const advExtrasVV = document.getElementById("advExtrasVV");
const vvVendedor = document.getElementById("vvVendedor");
const vvVendedorHelp = document.getElementById("vvVendedorHelp");

/* CAPACITACIONES */
const advExtrasCAP = document.getElementById("advExtrasCAP"); // Instructor (SOLO en Desglose)
const capInstructor = document.getElementById("capInstructor");
const capInstructorHelp = document.getElementById("capInstructorHelp");
const advExtrasCAP_FILTROS = document.getElementById("advExtrasCAP_FILTROS"); // Filtros CAP (Por cliente + estatus)
const capPorCliente = document.getElementById("capPorCliente");
const capClienteWrap = document.getElementById("capClienteWrap");
const capCliente = document.getElementById("capCliente");
const advExtrasCAP_ASIST = document.getElementById("advExtrasCAP_ASIST"); // Orden (SOLO en Asistencias)
const capOrden = document.getElementById("capOrden");

/* REPARTO → NUEVO: contenedor y checkboxes (HTML fijo en el template) */
const advExtrasREP = document.getElementById("advExtrasREP");
const repPendientesAsignar = document.getElementById("repPendientesAsignar");            // name="rep_pendientes_asignar"
const repAsignadas = document.getElementById("repAsignadas");                              // name="rep_asignadas"
const repAsignadasPorRepartidor = document.getElementById("repAsignadasPorRepartidor");    // name="rep_asignadas_por_repartidor"
const repDevueltosAlmacen = document.getElementById("repDevueltosAlmacen");                // name="rep_devueltos_almacen"
const repEntregadas = document.getElementById("repEntregadas");                            // name="rep_entregadas"

const comprasExtrasCard = document.getElementById("comprasExtrasCard");
const fechaIni = document.getElementById("fechaIni");
const fechaFin = document.getElementById("fechaFin");
const alertFechas = document.getElementById("alertFechas");

/* MOVIMIENTOS extras */
const advExtrasMOV = document.getElementById("advExtrasMOV");
const movPorUsuarioWrap = document.getElementById("movPorUsuarioWrap");
const movPorDeptoWrap = document.getElementById("movPorDeptoWrap");
const movPorAccionWrap = document.getElementById("movPorAccionWrap");
const movUsuario = document.getElementById("movUsuario");
const movUsuarioHelp = document.getElementById("movUsuarioHelp");
const movDepto = document.getElementById("movDepto");
const movDeptoHelp = document.getElementById("movDeptoHelp");
const movAccion = document.getElementById("movAccion");

/* =====================================
HELPERS UI (muestra/oculta)
===================================== */
function toggleAdvancedOptions(show) {
  advCard.classList.toggle("d-none", !show);
  setDisabledWithin(advCard, !show);
}
function setAdvancedChecksVisible(visible) {
  optsAvanzadas.classList.toggle("d-none", !visible);
  setDisabledWithin(optsAvanzadas, !visible);
}
function enableComprasMode(enabled) {
  comprasExtrasCard.classList.toggle("d-none", !enabled);
  setDisabledWithin(comprasExtrasCard, !enabled);
  toggleAdvancedOptions(!enabled);
}

function showCapFechaAgenda(show) {
  if (!capFechaAgendaWrap || !capFechaAgenda) return;
  capFechaAgendaWrap.classList.toggle("d-none", !show);
  capFechaAgenda.disabled = !show;
  if (!show) capFechaAgenda.checked = false;
}

function showVentasFiltrosExtras(show) {
  if (!advExtrasVENTAS_FILTROS) return;
  advExtrasVENTAS_FILTROS.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasVENTAS_FILTROS, !show);
  if (!show) {
    if (venEstatus) venEstatus.value = "";
    if (venTransito) venTransito.value = "";
  }
}

function showPipcExtras(show) {
  advExtrasPIPC.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasPIPC, !show);
  setAdvancedChecksVisible(!show);
  if (!show) {
    advExtrasPIPC.querySelectorAll('input[type="checkbox"]').forEach(cb => { cb.checked = false; });
  }
}

function showVppExtras(show) {
  advExtrasVPP.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasVPP, !show);
  if (!show) {
    if (vppSku) vppSku.value = "";
    if (vppAlmacen) vppAlmacen.value = "";
    if (vppUltima) vppUltima.checked = false;
  }
}

function setVppSkuVisible(visible) {
  if (vppSkuWrap) vppSkuWrap.classList.toggle('d-none', !visible);
  if (vppSku) {
    vppSku.disabled = !visible;
    if (!visible) vppSku.value = '';
  }
}

function setVppUltimaVisible(visible) {
  if (vppUltimaWrap) vppUltimaWrap.classList.toggle('d-none', !visible);
  if (vppUltima) {
    vppUltima.disabled = !visible;
    if (!visible) vppUltima.checked = false;
  }
}

function showTraspasosExtras(show) {
  advExtrasTR.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasTR, !show);
  if (!show && traspAlmacen) traspAlmacen.value = "";
}
function showAlmInventarioExtras(show) {
  advExtrasALM_INV.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasALM_INV, !show);
  if (!show && invAlmacen) invAlmacen.value = "";
}
function showVendedoresExtras(show) {
  advExtrasVV.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasVV, !show);
  if (!show && vvVendedor) vvVendedor.value = "";
}
/* Instructor (CAP → SOLO Desglose) */
function showCapacitacionesExtras(show) {
  advExtrasCAP.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasCAP, !show);
  if (!show && capInstructor) capInstructor.value = "";
}
/* Filtros CAP (Por cliente + estatus) → visibles cuando depto=Capacitaciones */
function showCapFiltrosExtras(show) {
  advExtrasCAP_FILTROS.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasCAP_FILTROS, !show);
  if (!show) {
    if (capPorCliente) capPorCliente.checked = false;
    if (capClienteWrap) capClienteWrap.classList.add("d-none");
    if (capCliente) { capCliente.disabled = true; capCliente.value = ""; }
    ["capRealizados","capPendientes","capSinRevisar","capPospuestos","capPagados"]
      .forEach(id => {
        const el = document.getElementById(id);
        if (el) el.checked = false;
      });
  }
}
/* Asistencias CAP → campo Orden SOLO cuando reporte = Asistencias */
function showCapAsistenciaExtras(show) {
  advExtrasCAP_ASIST.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasCAP_ASIST, !show);
  if (!show && capOrden) capOrden.value = "";
}

/* =====================================
MOV: helpers de UI
===================================== */
function showMovExtras(show) {
  if (!advExtrasMOV) return;
  advExtrasMOV.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasMOV, !show);

  if (!show) {
    showMovPorUsuario(false);
    showMovPorDepto(false);
    showMovPorAccion(false);
    if (movUsuario) movUsuario.value = "";
    if (movDepto) movDepto.value = "";
    if (movAccion) movAccion.value = "";
  }
}
function showMovPorUsuario(show) {
  if (!movPorUsuarioWrap) return;
  movPorUsuarioWrap.classList.toggle("d-none", !show);
  setDisabledWithin(movPorUsuarioWrap, !show);
  if (!show && movUsuario) movUsuario.value = "";
}
function showMovPorDepto(show) {
  if (!movPorDeptoWrap) return;
  movPorDeptoWrap.classList.toggle("d-none", !show);
  setDisabledWithin(movPorDeptoWrap, !show);
  if (!show && movDepto) movDepto.value = "";
}
function showMovPorAccion(show) {
  if (!movPorAccionWrap) return;
  movPorAccionWrap.classList.toggle("d-none", !show);
  setDisabledWithin(movPorAccionWrap, !show);
  if (!show && movAccion) movAccion.value = "";
}

/* =====================================
REPARTO: helpers de UI (NUEVO)
===================================== */
function showRepartoExtras(show) {
  if (!advExtrasREP) return;
  advExtrasREP.classList.toggle("d-none", !show);
  setDisabledWithin(advExtrasREP, !show);
  if (!show) {
    [repPendientesAsignar, repAsignadas, repAsignadasPorRepartidor, repDevueltosAlmacen, repEntregadas]
      .forEach(cb => { if (cb) cb.checked = false; });
  }
}

/* =====================================
RENDER UI
===================================== */
function renderCategorias() {
  categoriasWrap.innerHTML = "";
  CATEGORIAS.forEach((cat, idx) => {
    const catIdRaw = cat?.id ?? ('local-' + (idx + 1));
    const catNombre = String(cat?.nombre ?? ('Categoría ' + (idx + 1)));
    const htmlId = `cat-${catIdRaw}`;

    const input = document.createElement("input");
    input.type = "checkbox";
    input.className = "btn-check";
    input.id = htmlId;
    input.name = "categorias[]";
    input.value = catIdRaw;
    input.setAttribute('data-name', catNombre);
    input.autocomplete = "off";

    const label = document.createElement("label");
    label.className = "btn btn-outline-primary btn-sm";
    label.setAttribute("for", htmlId);
    label.textContent = catNombre;
    label.title = catNombre;

    categoriasWrap.appendChild(input);
    categoriasWrap.appendChild(label);
  });
}

function getAllowedAdvancedSet(endpointKey) {
  if (!endpointKey || !(endpointKey in ADVANCED_RULES)) return null;
  return new Set(ADVANCED_RULES[endpointKey] || []);
}

function renderOpcionesAvanzadas(permitidas = null, endpointKey = null) {
  optsAvanzadas.innerHTML = "";

  const allowedSet = getAllowedAdvancedSet(endpointKey);
  const defaults = Array.isArray(DEFAULTS_BY_ENDPOINT?.[endpointKey])
    ? new Set(DEFAULTS_BY_ENDPOINT[endpointKey])
    : new Set();

  ADVANCED_OPTIONS_MASTER.forEach((opt) => {
    const id = "adv_" + opt.name;
    const isAllowed = (allowedSet === null) ? true : allowedSet.has(opt.name);
    const isDefault = isAllowed && defaults.has(opt.name);

    const col = document.createElement("div");
    col.className = "col-12 col-md-4";

    const div = document.createElement("div");
    div.className = "form-check form-switch";

    const input = document.createElement("input");
    input.type = "checkbox";
    input.className = "form-check-input";
    input.id = id;
    input.name = opt.name;
    input.value = "1";
    input.dataset.allowed = isAllowed ? "1" : "0";

    const label = document.createElement("label");
    label.className = "form-check-label";
    label.setAttribute("for", id);
    label.textContent = opt.label;

    if (!isAllowed) {
      input.checked = false;
      input.disabled = true;
      input.tabIndex = -1;
      div.classList.add("adv-opt-disabled");
      label.classList.add("pe-none");
      label.title = "No aplica para este reporte";
    } else {
      input.checked = !!isDefault;
      if (isDefault) {
        label.title = "Recomendado para este reporte";
        label.setAttribute("aria-label", "Opción recomendada");
      }
    }

    div.appendChild(input);
    div.appendChild(label);
    col.appendChild(div);
    optsAvanzadas.appendChild(col);
  });

  if (allowedSet !== null && (permitidas?.length === 0)) {
    avisoOpciones.textContent = "Este reporte no admite opciones avanzadas.";
  } else if (allowedSet !== null) {
    avisoOpciones.textContent = "Solo se permiten las opciones aplicables. Algunas vienen activadas como recomendación.";
  } else {
    avisoOpciones.textContent = "Activa o desactiva detalles a incluir en el PDF.";
  }
}

/* Bloqueo extra por delegación: evita toggles en opciones no permitidas */
optsAvanzadas.addEventListener("click", (e) => {
  const lbl = e.target.closest("label.form-check-label");
  if (lbl) {
    const forId = lbl.getAttribute("for");
    if (forId) {
      const cb = document.getElementById(forId);
      if (cb && cb.dataset.allowed === "0") {
        e.preventDefault();
        e.stopPropagation();
        return;
      }
    }
  }
  const cbx = e.target.closest('input[type="checkbox"]');
  if (cbx && cbx.dataset.allowed === "0") {
    e.preventDefault();
    e.stopPropagation();
    cbx.checked = false;
  }
}, true);

/* Sanitizar antes del submit: nunca mandar opciones bloqueadas */
function sanitizeAdvancedBeforeSubmit(endpointKey) {
  const allowedSet = getAllowedAdvancedSet(endpointKey);
  if (allowedSet === null) return;
  optsAvanzadas.querySelectorAll('input[type="checkbox"]').forEach(cb => {
    if (!allowedSet.has(cb.name)) {
      cb.checked = false;
      cb.disabled = true;
    }
  });
}

/* =====================================
CATEGORÍAS - Políticas por departamento
===================================== */
const DEPTS_CON_CATS = new Set(["Almacen", "Ventas", "Tienda"]);

function hideCategoriesCard() {
  categoriasCard.classList.add("d-none");
  categoriasWrap.querySelectorAll('input[name="categorias[]"]').forEach(cb => { cb.disabled = true; cb.checked = false; });
}
function showCategoriesCard() {
  categoriasCard.classList.remove("d-none");
  categoriasWrap.querySelectorAll('input[name="categorias[]"]').forEach(cb => { cb.disabled = false; });
}
function enforceOnlyPIPCCategory() {
  categoriasCard.classList.add("d-none");
  let pipcFound = false;

  categoriasWrap.querySelectorAll('input[name="categorias[]"]').forEach(cb => {
    const n = normalizeBasic(cb.getAttribute('data-name') || '');
    if (n === 'pipc') {
      cb.disabled = false;
      cb.checked = true;
      pipcFound = true;
    } else {
      cb.checked = false;
      cb.disabled = true;
    }
  });

  if (!pipcFound) {
    categoriasWrap.querySelectorAll('input[name="categorias[]"]').forEach(cb => { cb.checked = false; cb.disabled = true; });
  }
}
function updateCategoriasPolicy(currentDepto) {
  if (DEPTS_CON_CATS.has(currentDepto)) {
    showCategoriesCard();
  } else if (currentDepto === "PIPC") {
    enforceOnlyPIPCCategory();
  } else {
    hideCategoriesCard();
  }
}

/* =====================================
SELECTS dependientes por reporte
===================================== */
function updateTiposPorDepartamento(depto) {
  selReporte.innerHTML = "";
  selReporte.disabled = true;

  // Cuando es Reparto, se usa el bloque de checkboxes; no llenar el select
  if (depto === "Reparto" || depto === "PIPC") {
    selReporte.innerHTML = `<option value="" disabled selected>— No aplica —</option>`;
    return;
  }

  const lista = REPORTES[depto] || [];
  if (!lista.length) {
    selReporte.innerHTML = `<option value="" disabled selected>— No hay reportes definidos —</option>`;
    return;
  }
  selReporte.disabled = false;
  selReporte.insertAdjacentHTML("beforeend", `<option value="" disabled selected>— Selecciona —</option>`);
  lista.forEach(r => {
    const opt = document.createElement("option");
    opt.value = r.endpoint;
    opt.textContent = r.label;
    selReporte.appendChild(opt);
  });
}

function setFormAction(endpoint) {
  const finalUrl = "./pdfreportes/" + endpoint;
  form.action = finalUrl;
  actionPreview.textContent = finalUrl;
}

/* =====================================
VALIDACIONES
===================================== */
function validarFechas() {
  alertFechas.classList.add("d-none");
  const ini = fechaIni.value;
  const fin = fechaFin.value;
  if (ini && fin && fin < ini) {
    alertFechas.classList.remove("d-none");
    return false;
  }
  return true;
}
function validarBootstrap(e) {
  if (!form.checkValidity()) {
    e.preventDefault();
    e.stopPropagation();
  }
  form.classList.add('was-validated');
}

/* =====================================
AJAX: Categorías (POST hook/action)
===================================== */
async function loadCategoriasFromAPI() {
  var hook = 'codi_hook', action = 'load';
  categoriasNotice.textContent = 'Cargando categorías desde el servidor...';

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_categorias_json', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();

    let arr = [];
    if (Array.isArray(data)) arr = data;
    else if (data && Array.isArray(data.data)) arr = data.data;
    else if (data && Array.isArray(data.categorias)) arr = data.categorias;
    else if (data && typeof data === 'object' && 'catId' in data && 'catNombre' in data) arr = [data];

    const mapped = arr.map(item => {
      const id = item?.catId;
      const nombre = item?.catNombre;
      if (id == null || !nombre) return null;
      return { id, nombre: String(nombre).trim() };
    }).filter(Boolean);

    const unique = []; const seen = new Set();
    for (const c of mapped) {
      const k = String(c.id);
      if (!seen.has(k)) { seen.add(k); unique.push(c); }
    }

    if (unique.length) {
      CATEGORIAS.splice(0, CATEGORIAS.length, ...unique);
      renderCategorias();
      categoriasNotice.textContent = `Categorías cargadas (${unique.length}).`;
      updateCategoriasPolicy(selDepto.value || "");
    } else {
      categoriasNotice.textContent = 'No se recibieron categorías válidas; se usan las de ejemplo.';
      updateCategoriasPolicy(selDepto.value || "");
    }

  } catch (err) {
    console.error('Error cargando categorías:', err);
    categoriasNotice.textContent = 'No se pudieron cargar categorías (se usan las de ejemplo).';
    const warn = document.createElement('div');
    warn.className = 'alert alert-info mt-2';
    warn.textContent = 'No se pudieron cargar categorías desde el servidor, se usan las categorías por defecto.';
    categoriasWrap.parentNode.appendChild(warn);
    updateCategoriasPolicy(selDepto.value || "");
  }
}

/* =====================================
AJAX: VENTAS (Estatus y En tránsito)
===================================== */
let ESTATUS_ORDENES = [];
let estatusLoaded = false;

async function loadEstatusOrdenesFromAPI() {
  if (estatusLoaded) { renderIntoSelect(venEstatus, ESTATUS_ORDENES); return; }

  const params = new URLSearchParams();
  params.append('hook', 'codi_hook');
  params.append('action', 'load');

  if (venEstatus) { venEstatus.innerHTML = `<option value="">Cargando...</option>`; venEstatus.disabled = true; }

  try {
    const res = await fetch('https://grupoxtinfire.com/admin/ajax/codi_get_status', {
      method: 'POST',
      headers: { 'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString()
    });
    const data = await res.json();

    const arr = Array.isArray(data?.data) ? data.data : (Array.isArray(data) ? data : []);
    ESTATUS_ORDENES = arr
      .map(it => {
        const id = it?.IdEstatus ?? it?.id ?? it?.value ?? null;
        const nombre = (it?.Descripcion ?? it?.descripcion ?? it?.name ?? "").toString().trim();
        if (id == null || !nombre) return null;
        return { id, nombre };
      })
      .filter(Boolean);

    renderIntoSelect(venEstatus, ESTATUS_ORDENES);
    estatusLoaded = true;

  } catch(e) {
    if (venEstatus) { venEstatus.innerHTML = `<option value="">(No disponible)</option>`; venEstatus.disabled = true; }
  }
}

let TRANSITOS = [];
let transitoLoaded = false;

async function loadEstadoTransitoFromAPI() {
  if (transitoLoaded) { renderIntoSelect(venTransito, TRANSITOS); return; }

  const params = new URLSearchParams();
  params.append('hook', 'codi_hook');
  params.append('action', 'load');

  if (venTransito) { venTransito.innerHTML = `<option value="">Cargando...</option>`; venTransito.disabled = true; }

  try {
    const res = await fetch('https://grupoxtinfire.com/admin/ajax/codi_get_estado_transito', {
      method: 'POST',
      headers: { 'Content-Type':'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString()
    });
    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : (Array.isArray(data) ? data : []);
    TRANSITOS = arr
      .map(it => {
        const id = it?.IdDepto ?? it?.id ?? it?.value ?? null;
        const nombre = (it?.DeptoDescripcion ?? it?.descripcion ?? it?.name ?? "").toString().trim();
        if (id == null || !nombre) return null;
        return { id, nombre };
      })
      .filter(Boolean);

    renderIntoSelect(venTransito, TRANSITOS);
    transitoLoaded = true;

  } catch(e) {
    if (venTransito) { venTransito.innerHTML = `<option value="">(No disponible)</option>`; venTransito.disabled = true; }
  }
}

// Utilidad para rellenar selects
function renderIntoSelect(selectEl, items, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  (items || []).forEach(it => {
    const op = document.createElement('option');
    op.value = it.id; op.textContent = it.nombre;
    selectEl.appendChild(op);
  });
  selectEl.disabled = false;
}

/* =====================================
AJAX: Almacenes (POST hook/action)
===================================== */
let ALMACENES = [];
let almacenesLoaded = false;

function renderAlmacenesIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  ALMACENES.forEach(a => {
    const opt = document.createElement('option');
    opt.value = a.id; opt.textContent = a.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

async function loadAlmacenesFromAPI() {
  if (almacenesLoaded) {
    renderAlmacenesIntoSelect(vppAlmacen, true);
    renderAlmacenesIntoSelect(traspAlmacen, true);
    renderAlmacenesIntoSelect(invAlmacen, true);
    return;
  }

  var hook = 'codi_hook', action = 'load';

  [vppAlmacen, traspAlmacen, invAlmacen].forEach(sel => {
    if (sel) { sel.innerHTML = `<option value="">Cargando almacenes…</option>`; sel.disabled = true; }
  });

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_almacenes_json', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : [];

    ALMACENES = arr
      .filter(a => a && Number(a.almEstatus) === 1)
      .map(a => ({ id: a.almId, nombre: String(a.almNombre || '').trim() }))
      .filter(a => a.id != null && a.nombre);

    renderAlmacenesIntoSelect(vppAlmacen, true);
    renderAlmacenesIntoSelect(traspAlmacen, true);
    renderAlmacenesIntoSelect(invAlmacen, true);

    if (vppAlmacenHelp) vppAlmacenHelp.textContent = 'Selecciona el almacén donde buscar el producto.';
    if (traspAlmacenHelp) traspAlmacenHelp.textContent = 'Selecciona el almacén para el reporte de traspasos.';
    if (invAlmacenHelp) invAlmacenHelp.textContent = 'Selecciona el almacén para el reporte de inventarios.';

    almacenesLoaded = true;

  } catch (err) {
    console.error('Error cargando almacenes:', err);
    [vppAlmacen, traspAlmacen, invAlmacen].forEach(sel => {
      if (sel) { sel.innerHTML = `<option value="">(No disponible)</option>`; sel.disabled = true; }
    });
    if (invAlmacenHelp) invAlmacenHelp.textContent = 'No se pudieron cargar los almacenes. Intenta más tarde.';
  }
}

/* =====================================
AJAX: Vendedores (POST hook/action)
===================================== */
let VENDEDORES = [];
let vendedoresLoaded = false;

function renderVendedoresIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  (VENDEDORES || []).forEach(v => {
    const opt = document.createElement('option');
    opt.value = v.id; opt.textContent = v.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

async function loadVendedoresFromAPI() {
  if (vendedoresLoaded) { renderVendedoresIntoSelect(vvVendedor, true); return; }

  var hook = 'codi_hook', action = 'load';

  if (vvVendedor) {
    vvVendedor.innerHTML = `<option value="">Cargando vendedores…</option>`;
    vvVendedor.disabled = true;
  }

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_vendedores_json', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : [];

    VENDEDORES = arr
      .filter(it => it && it.IdUsuario != null && it.Usuario)
      .filter(it => {
        const d = normalizeBasic(it.DeptoDescripcion);
        return d === 'ventas' || d === 'tienda';
      })
      .map(it => ({ id: it.IdUsuario, nombre: String(it.Usuario).trim() }))
      .reduce((acc, cur) => {
        const k = String(cur.id);
        if (!acc.map.has(k)) { acc.map.set(k, true); acc.list.push(cur); }
        return acc;
      }, { map: new Map(), list: [] }).list
      .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));

    if (!VENDEDORES.length) {
      VENDEDORES = arr
        .filter(it => it && it.IdUsuario != null && it.Usuario)
        .map(it => ({ id: it.IdUsuario, nombre: String(it.Usuario).trim() }))
        .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));
    }

    renderVendedoresIntoSelect(vvVendedor, true);
    if (vvVendedorHelp) vvVendedorHelp.textContent = 'Selecciona el vendedor (incluye Ventas y Tienda).';
    vendedoresLoaded = true;

  } catch (err) {
    console.error('Error cargando vendedores:', err);
    if (vvVendedor) {
      vvVendedor.innerHTML = `<option value="">(No disponible)</option>`;
      vvVendedor.disabled = true;
    }
    if (vvVendedorHelp) vvVendedorHelp.textContent = 'No se pudieron cargar vendedores. Intenta más tarde.';
  }
}

/* =====================================
AJAX: Instructores (POST hook/action)
===================================== */
let INSTRUCTORES = [];
let instructoresLoaded = false;

function renderInstructoresIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  (INSTRUCTORES || []).forEach(p => {
    const opt = document.createElement('option');
    opt.value = p.id; opt.textContent = p.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

async function loadInstructoresFromAPI() {
  if (instructoresLoaded) { renderInstructoresIntoSelect(capInstructor, true); return; }

  var hook = 'codi_hook', action = 'load';

  if (capInstructor) {
    capInstructor.innerHTML = `<option value="">Cargando encargados…</option>`;
    capInstructor.disabled = true;
  }

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_instructores_json', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : [];

    INSTRUCTORES = arr
      .filter(it => it && it.IdUsuario != null && it.Usuario)
      .filter(it => normalizeBasic(it.DeptoDescripcion) === 'capacitacion')
      .map(it => ({ id: it.IdUsuario, nombre: String(it.Usuario).trim() }))
      .reduce((acc, cur) => {
        const k = String(cur.id);
        if (!acc.map.has(k)) { acc.map.set(k, true); acc.list.push(cur); }
        return acc;
      }, { map: new Map(), list: [] }).list
      .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));

    if (!INSTRUCTORES.length) {
      INSTRUCTORES = arr
        .filter(it => it && it.IdUsuario != null && it.Usuario)
        .map(it => ({ id: it.IdUsuario, nombre: String(it.Usuario).trim() }))
        .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));
    }

    renderInstructoresIntoSelect(capInstructor, true);
    if (capInstructorHelp) capInstructorHelp.textContent = 'Selecciona el encargado de capacitación.';
    instructoresLoaded = true;

  } catch (err) {
    console.error('Error cargando encargados:', err);
    if (capInstructor) {
      capInstructor.innerHTML = `<option value="">(No disponible)</option>`;
      capInstructor.disabled = true;
    }
    if (capInstructorHelp) capInstructorHelp.textContent = 'No se pudieron cargar encargados. Intenta más tarde.';
  }
}

/* =====================================
MOVIMIENTOS
===================================== */
let USUARIOS_REPORTE = [];
let usuariosReporteLoaded = false;

function renderUsuariosIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  (USUARIOS_REPORTE || []).forEach(u => {
    const opt = document.createElement('option');
    opt.value = u.id; opt.textContent = u.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

async function loadUsuariosReporteFromAPI() {
  if (usuariosReporteLoaded) { renderUsuariosIntoSelect(movUsuario, true); return; }

  var hook = 'codi_hook', action = 'load';

  if (movUsuario) {
    movUsuario.innerHTML = `<option value="">Cargando usuarios…</option>`;
    movUsuario.disabled = true;
  }

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_usuarios_reporte', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : [];

    USUARIOS_REPORTE = arr
      .filter(it => it && it.IdUsuario != null && it.Usuario && Number(it.activo ?? 1) === 1)
      .map(it => ({ id: it.IdUsuario, nombre: String(it.Usuario).trim() }))
      .reduce((acc, cur) => {
        const k = String(cur.id);
        if (!acc.map.has(k)) { acc.map.set(k, true); acc.list.push(cur); }
        return acc;
      }, { map: new Map(), list: [] }).list
      .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));

    renderUsuariosIntoSelect(movUsuario, true);
    if (movUsuarioHelp) movUsuarioHelp.textContent = 'Selecciona el usuario para el reporte.';
    usuariosReporteLoaded = true;

  } catch (err) {
    console.error('Error cargando usuarios-reporte:', err);
    if (movUsuario) {
      movUsuario.innerHTML = `<option value="">(No disponible)</option>`;
      movUsuario.disabled = true;
    }
    if (movUsuarioHelp) movUsuarioHelp.textContent = 'No se pudieron cargar usuarios. Intenta más tarde.';
  }
}

let DEPARTAMENTOS = [];
let departamentosLoaded = false;

function renderDepartamentosIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todos —</option>` : '';
  (DEPARTAMENTOS || []).forEach(d => {
    const opt = document.createElement('option');
    opt.value = d.id; opt.textContent = d.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

async function loadDepartamentosFromAPI() {
  if (departamentosLoaded) { renderDepartamentosIntoSelect(movDepto, true); return; }

  var hook = 'codi_hook', action = 'load';

  if (movDepto) {
    movDepto.innerHTML = `<option value="">Cargando departamentos…</option>`;
    movDepto.disabled = true;
  }

  try {
    const params = new URLSearchParams();
    params.append('hook', hook);
    params.append('action', action);

    const res = await fetch('ajax/codi_get_departamento_json', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8' },
      body: params.toString(),
      mode: 'cors',
      credentials: 'omit'
    });
    if (!res.ok) throw new Error('HTTP ' + res.status);

    const data = await res.json();
    const arr = Array.isArray(data?.data) ? data.data : [];

    DEPARTAMENTOS = arr
      .filter(it => it && it.IdDepto != null && it.DeptoDescripcion && Number(it.Estatus ?? 1) === 1)
      .map(it => ({ id: it.IdDepto, nombre: String(it.DeptoDescripcion).trim() }))
      .sort((a, b) => a.nombre.localeCompare(b.nombre, 'es', { sensitivity: 'base' }));

    renderDepartamentosIntoSelect(movDepto, true);
    if (movDeptoHelp) movDeptoHelp.textContent = 'Selecciona el departamento para el reporte.';
    departamentosLoaded = true;

  } catch (err) {
    console.error('Error cargando departamentos:', err);
    if (movDepto) {
      movDepto.innerHTML = `<option value="">(No disponible)</option>`;
      movDepto.disabled = true;
    }
    if (movDeptoHelp) movDeptoHelp.textContent = 'No se pudieron cargar departamentos. Intenta más tarde.';
  }
}

/* =====================================
MOV: Acciones (local)
===================================== */
const MOV_ACCIONES = [
  { id: 'login', nombre: 'Inicio de sesión' },
  { id: 'logout', nombre: 'Cierre de sesión' },
  { id: 'crear', nombre: 'Crear' },
  { id: 'actualizar', nombre: 'Actualizar' },
  { id: 'eliminar', nombre: 'Eliminar' },
  { id: 'visualizar', nombre: 'Visualizar' },
  { id: 'imprimir', nombre: 'Imprimir' },
  { id: 'exportar', nombre: 'Exportar' },
  { id: 'aprobar', nombre: 'Aprobar' },
  { id: 'rechazar', nombre: 'Rechazar' }
];

function renderAccionesIntoSelect(selectEl, includeTodos = true) {
  if (!selectEl) return;
  selectEl.innerHTML = includeTodos ? `<option value="">— Todas —</option>` : '';
  MOV_ACCIONES.forEach(a => {
    const opt = document.createElement('option');
    opt.value = a.id; opt.textContent = a.nombre;
    selectEl.appendChild(opt);
  });
  selectEl.disabled = false;
}

/* =====================================
CAP: handlers únicos (no crear nodos)
===================================== */
function setupCapPorClienteHandlerOnce() {
  if (!capPorCliente || capPorCliente.dataset.bound) return;
  capPorCliente.dataset.bound = "1";
  capPorCliente.addEventListener("change", () => {
    const on = capPorCliente.checked;
    if (capClienteWrap) capClienteWrap.classList.toggle("d-none", !on);
    if (capCliente) {
      capCliente.disabled = !on;
      if (!on) capCliente.value = "";
    }
  });
  // estado inicial
  if (capCliente) capCliente.disabled = true;
  if (capClienteWrap) capClienteWrap.classList.add("d-none");
}

/* =====================================
EVENTOS
===================================== */
selDepto.addEventListener("change", async () => {
  const depto = selDepto.value;

  updateCategoriasPolicy(depto);
  updateTiposPorDepartamento(depto);
  setFormAction("pdfcatalogo");

  // Reset general
  renderOpcionesAvanzadas(null, null);
  setAdvancedChecksVisible(true);
  enableComprasMode(false);
  showVppExtras(false);
  setVppSkuVisible(false);
  showTraspasosExtras(false);
  showAlmInventarioExtras(false);
  showVendedoresExtras(false);
  showMovExtras(false);
  showVentasFiltrosExtras(false);
  showRepartoExtras(false);
  showCapFechaAgenda(false);

  // CAP: Mostrar sus filtros, ocultar instructor y campo Orden por defecto
  if (depto === "Capacitaciones") {
    setAdvancedChecksVisible(false);
    showCapFiltrosExtras(true);
    setupCapPorClienteHandlerOnce();
    showCapacitacionesExtras(false); // instructor solo cuando sea Desglose
    showCapAsistenciaExtras(false);  // Orden solo cuando sea Asistencias
    showCapFechaAgenda(false);
  } else {
    showCapFiltrosExtras(false);
    showCapacitacionesExtras(false);
    showCapAsistenciaExtras(false);
    showCapFechaAgenda(false);
  }

  // === PIPC: ocultar "Tipo de reporte" y usar checkboxes propios ===
  if (depto === "PIPC") {
    if (selReporteCol) selReporteCol.classList.add("d-none");
    selReporte.disabled = true;
    selReporte.required = false;

    toggleAdvancedOptions(true);
    showPipcExtras(true);
    setFormAction(PIPC_ENDPOINT);
    return; // evita tocar más toggles
  } else {
    showPipcExtras(false);
  }

  // === REPARTO: ocultar "Tipo de reporte" y mostrar sus checkboxes en Opciones avanzadas ===
  if (depto === "Reparto") {
    if (selReporteCol) selReporteCol.classList.add("d-none");
    selReporte.disabled = true;
    selReporte.required = false;

    toggleAdvancedOptions(true);
    setAdvancedChecksVisible(false); // ocultar los masters estándar
    showRepartoExtras(true);
    setFormAction(REPARTO_ENDPOINT);
  } else {
    showRepartoExtras(false);
    if (selReporteCol) selReporteCol.classList.remove("d-none");
    selReporte.disabled = false;
    selReporte.required = true;
  }
});

selReporte.addEventListener("change", async () => {
  const endpoint = selReporte.value;
  setFormAction(endpoint);

  const depto = selDepto.value;

  /* ===== Capacitaciones: lógica propia ===== */
  if (depto === "Capacitaciones") {
    // Mantener visibles los filtros CAP
    showCapFiltrosExtras(true);
    setupCapPorClienteHandlerOnce();

    const isDesglose    = (endpoint === CAP_ENDPOINTS.DESGLOSE);
    const isAsistencias = (endpoint === CAP_ENDPOINTS.ASISTENCIAS);

    // Checkbox "Buscar por fecha agendada" solo en Desglose
    showCapFechaAgenda(isDesglose);

    // Instructor SOLO en Desglose
    showCapacitacionesExtras(isDesglose);
    if (isDesglose && !instructoresLoaded) {
      await loadInstructoresFromAPI();
    }

    // Campo Orden SOLO en Asistencias
    showCapAsistenciaExtras(isAsistencias);

    // No usamos las opciones avanzadas estándar en CAP
    return;
  }

  /* ===== Resto de departamentos ===== */
  const isCompras            = (depto === "Almacen" && endpoint === "pdfcompras");
  const isVentasPorProducto  = (depto === "Ventas" && endpoint === "pdfVentasPorProducto");
  const isVentasPorCategoria = (depto === "Ventas" && endpoint === "pdfVentasPorCategoria");
  const isVentasGenerales    = (depto === "Ventas" && endpoint === "pdfVentasGenerales");
  const isAlmacenTraspasos   = (depto === "Almacen" && endpoint === "pdftraspasos");
  const isVentasPorVendedor  = (depto === "Ventas" && endpoint === "pdfVentasPorVendedor");
  const isAlmacenInventarios = (depto === "Almacen" && endpoint === "pdfinvetarios");

  const isMovDepto           = (depto === "Movimientos por usuario");
  const isMovPorUsuario      = isMovDepto && endpoint === "mov-usuarios-por-usuario";
  const isMovPorDepartamento = isMovDepto && endpoint === "mov-usuarios-por-departamento";
  const isMovPorAccion       = isMovDepto && endpoint === "mov-usuarios-por-accion";

  // Opciones avanzadas estándar
  if (!isCompras) {
    const permitidas = ADVANCED_RULES[endpoint] ?? [];
    renderOpcionesAvanzadas(permitidas, endpoint);
  }
  setAdvancedChecksVisible(!isCompras);

  enableComprasMode(isCompras);

  // Ventas → Producto / Categoría
  showVppExtras(isVentasPorProducto || isVentasPorCategoria);
  setVppSkuVisible(isVentasPorProducto);
  if ((isVentasPorProducto || isVentasPorCategoria) && !almacenesLoaded) {
    await loadAlmacenesFromAPI();
  }
  setVppUltimaVisible(isVentasPorProducto);

  // Ventas → filtros nuevos
  const mustShowVentasFiltros = (isVentasGenerales || isVentasPorVendedor);
  showVentasFiltrosExtras(mustShowVentasFiltros);
  if (mustShowVentasFiltros) {
    if (!estatusLoaded) await loadEstatusOrdenesFromAPI(); else renderIntoSelect(venEstatus, ESTATUS_ORDENES);
    if (!transitoLoaded) await loadEstadoTransitoFromAPI(); else renderIntoSelect(venTransito, TRANSITOS);
  }

  // Almacén → Traspasos
  showTraspasosExtras(isAlmacenTraspasos);
  if (isAlmacenTraspasos && !almacenesLoaded) await loadAlmacenesFromAPI();

  // Almacén → Inventarios
  showAlmInventarioExtras(isAlmacenInventarios);
  if (isAlmacenInventarios && !almacenesLoaded) await loadAlmacenesFromAPI();

  // Ventas → Por vendedor
  showVendedoresExtras(isVentasPorVendedor);
  if (isVentasPorVendedor && !vendedoresLoaded) await loadVendedoresFromAPI();

  // Movimientos
  showMovExtras(isMovDepto && (isMovPorUsuario || isMovPorDepartamento || isMovPorAccion));

  showMovPorUsuario(isMovPorUsuario);
  if (isMovPorUsuario && !usuariosReporteLoaded) await loadUsuariosReporteFromAPI();
  else if (isMovPorUsuario) renderUsuariosIntoSelect(movUsuario, true);

  showMovPorDepto(isMovPorDepartamento);
  if (isMovPorDepartamento && !departamentosLoaded) await loadDepartamentosFromAPI();
  else if (isMovPorDepartamento) renderDepartamentosIntoSelect(movDepto, true);

  showMovPorAccion(isMovPorAccion);
  if (isMovPorAccion) renderAccionesIntoSelect(movAccion, true);

  // Asegura ocultar restos de CAP si vienen de cambiar depto
  showCapFiltrosExtras(false);
  showCapacitacionesExtras(false);
  showCapAsistenciaExtras(false);
  showCapFechaAgenda(false);
});

fechaIni.addEventListener("change", validarFechas);
fechaFin.addEventListener("change", validarFechas);

form.addEventListener("submit", (e) => {
  const depto = selDepto.value;
  const isNoSelectDept = (depto === "PIPC" || depto === "Reparto"); // aquí ya no exigimos selReporte

  // Endpoint para sanitización (en PIPC y Reparto no hay selReporte activo)
  const endpointKey = isNoSelectDept ? (depto === "PIPC" ? PIPC_ENDPOINT : REPARTO_ENDPOINT) : (selReporte.value || null);
  sanitizeAdvancedBeforeSubmit(endpointKey);

  // Validaciones
  if (!validarFechas()) { e.preventDefault(); e.stopPropagation(); return; }
  validarBootstrap(e);

  if (!depto) { e.preventDefault(); e.stopPropagation(); selDepto.focus(); return; }

  if (!isNoSelectDept && !selReporte.value) {
    e.preventDefault(); e.stopPropagation(); selReporte.focus(); return;
  }
});

/* =====================================
INIT
===================================== */
(function init(){
  renderCategorias();
  loadCategoriasFromAPI();
  updateCategoriasPolicy(selDepto.value || "");
  renderOpcionesAvanzadas(null, null);
  setFormAction("pdfcatalogo");
  enableComprasMode(false);
  setAdvancedChecksVisible(true);

  // Ocultar todos los extras al inicio
  showVppExtras(false);
  setVppSkuVisible(false);
  showTraspasosExtras(false);
  showAlmInventarioExtras(false);
  showVendedoresExtras(false);
  showMovExtras(false);
  showVentasFiltrosExtras(false);
  showCapFiltrosExtras(false);
  setupCapPorClienteHandlerOnce();
  showCapacitacionesExtras(false);
  showCapAsistenciaExtras(false);
  showCapFechaAgenda(false);
  showRepartoExtras(false);
})();
